/*  Проект: измеритель проводимости
 *  author: Vladimir L. (aka Volume at the https://forum.homedistiller.ru/)
 *  Created: 08 Sep 2021
 *  Сигнал измерения реализован на таймер2 с выдачей на пин OC2A, ток проводимости меряет ADC с опорным 1.1В
 *
 *  2 предела, переключаются автоматически
 *  Проводимость отображается в десятых долях микросименс, т.е. 10 - это 1 мксименс
 *  
 *  добавлена опция для сокращения занимаемой памяти (под процессор 168)
 *==================================================================================================
 ************  правки **********************
 *  2025-10-30 - исправлен баг для параллельного подключения дисплея, вызывался init вместо begin
 */

// Выключает температурный сенсор и не будет выдавать в СОМ-порт инфу о командах и дамп
// при компиляции на процессоре 168 - раскомментируй, это сократит память программ с 16 до 12 кб
#define REDUCE_SIZE

#ifndef REDUCE_SIZE    
  #include <Wire.h>
  #include <OneWire.h>
  #include <DallasTemperature.h>
#endif
//--------конфигурация--------
//--------кол-во измерений для усреднения
#define SHUNT_ADC_MEASURES 256U
#define OUT_ADC_MEASURES 64U
//--------пауза между измерениями, мс
#define SLEEP_PAUSE_MS     1200l
#define OUT_WAKEUP_TIME_MS  120l
//--------лимиты переключения диапазона измерения, в разрядах ADC
#define LOW_LIMIT_RANGE 36
#define TOP_LIMIT_RANGE 157
// максимальное значение проводимости (при КЗ)
#define MAX_CONDUCTIVITY 9999999l
//-------- номинал ограничительного резистора
#define RPROTECT_NOMINAL_OHM  240
//-------- номинал резистора шунта с поправкой на вх.сопротивление ADC
//диапазон0 высоких проводимостей
#define SHUNT0_NOMINAL_OHM  49317
//диапазон1 низких проводимостей
#define SHUNT1_NOMINAL_OHM  51282
//--------множитель показаний проводимости
#define CONDUCTIVITY_MULTI 1.0
//--------настройки аналогового выхода
//--------коэф. для расчета U аналогового выхода f(x)=(1 - 1/(COEFF*x+1))
//   значение 0.003 дает в диапазоне проводимостей 0..20мкс почти линейный рост 0..1.9В
#define AN_OUT_CALCULATE_COEFF 0.003
// максимальный % Uп на аналоговом выходе
#define MAX_PRC_AOUT 99
//  максимальное значение счетчика таймера1. Определяет частоту а.выхода и разрядность ШИМ
//  по умолчанию задано 12 бит, частота при этом 16МГц/4096=4кГц
#define TOP_TIMER1 0x0FFFu

// номинал резистора, который будет подключаться вместо датчика при калибровке кнопкой
#define CALIBRATION_RESISTOR 82000l
#define OUT_DIVIDER 0.943

//период выдачи показаний в uart
#define PERIOD_SERIAL_MS   1000

//-----------PINS----------------------
// 1wire port, for 18DS20 PINB0
#define ONE_WIRE_BUS 8
// аналоговый выход (PWM Т1/OCA1)
#define PIN_ANOUT  PINB1 //D9
//кнопка "предел"
#define PIN_BUTTON_LIMIT   PINB2 //D10
//выход накачки
#define PIN_OUT   PINB3 //D11, OC2A
//кнопка "калибровка"
#define PIN_BUTTON_CLBR   PINB4 //D12
// выход ключа шунта //D13
#define PIN_SWITCH   PINB5

//--------канал ADC измерения
#define ADC_SHUNT 0
#define ADC_OUT 1

// поддерживаемые модели дисплея
#define LCD1602 0
#define LCD1602_I2C_PCF8574 2

// включить поддержку дисплея
#define DISPLAY_ENABLE
// твоя модель дисплея. Д.быть раскомментирована только одна строчка!
#define DISPLAY_MODEL LCD1602
//#define DISPLAY_MODEL LCD1602_I2C_PCF8574
//-------------------------------------
#ifdef DISPLAY_ENABLE
  #if DISPLAY_MODEL == LCD1602
      // пины подключения дисплея
      #define RS_PIN 2  // Пин сигнала RES дисплея
      #define E_PIN  3  // Пин сигнала EN дисплея
      #define D4_PIN 4  // Пин полубайта данных дисплея
      #define D5_PIN 5  // Пин полубайта данных дисплея
      #define D6_PIN 6  // Пин полубайта данных дисплея
      #define D7_PIN 7  // Пин полубайта данных дисплея
  #endif
  #if DISPLAY_MODEL == LCD1602_I2C_PCF8574
// ---!!!---по I2C дисплей подключается к пинам SCL SDA аппаратного I2C, на нано/промини это A4,A5
//          инициализировать A4,A5 не надо - это делает библиотека
  #endif
#endif

#define IS_RANGE0     (DDRB & _BV(PIN_SWITCH))
#define SET_RANGE1    (DDRB &= ~(_BV(PIN_SWITCH)))
#define SET_RANGE0    (DDRB |= _BV(PIN_SWITCH))

#define CALC_AVG_ADC(A)   ((measures.summ+(A)/2)/(A))
#define MEASURE_FINISHED  (measures.count==0)
#define IS_ADC_CHL_SHUNT  ((ADMUX&0x0F)==ADC_SHUNT)

//===================EEPROM адреса
#define EEPROM_VERSION    0x16
#define EEPROM_NEXT_ADR(A) (A+2)
// байт версии конфигурации
#define EEPROM_ADR_VERSION  (uint8_t*)0
// адрес записи конфигурации
#define EEPROM_ADR_CFG (void*)EEPROM_NEXT_ADR(EEPROM_ADR_VERSION)

//проверка выхода накачки
#define IS_OUTPUT_ON    (DDRB & _BV(PIN_OUT))
//выкл выход накачки
#define OUTPUT_OFF      (DDRB &= ~(_BV(PIN_OUT)))
//вкл выход накачки
#define OUTPUT_ON       (DDRB |= _BV(PIN_OUT))

//--------частота накачки
#define OUTPUT_FREQ 1020L
//--частота ядра
#define FCPU 16000000L
//--делитель таймера2
#define T2_DIV 128
//--кол-во тиков таймера2 в полупериод накачки
#define SEMIPERIOD_TICKS      (uint8_t)(((FCPU/2)/OUTPUT_FREQ)/T2_DIV)
//--задержка старта АЦП от фронта накачки, в тиках таймера2
#define ADC_START_DELAY_TICKS (uint8_t)( ((ADC_MEAS_TICKS+ADC_MEASURE_GAP)*ADC_DIV)/T2_DIV)

#define INT_OFF do {ADCSRA &= ~(_BV(ADIE)); TIMSK2 &= ~(_BV(OCIE2B));} while(0)
#define INT_ON do {ADCSRA |= _BV(ADIE); TIMSK2 |= _BV(OCIE2B);} while(0)
//--------отладка
//#define DEBUG
#ifdef DEBUG
#define DBG(A)   Serial.print((A))
#define DBGLN(A) Serial.println((A))
#else
#define DBG(A)
#define DBGLN(A)
#endif

typedef enum {
  range_mode_Auto,
  range_mode_0R,
  range_mode_1R
} range_mode_t;

typedef enum {
  clbr_NO,
  clbr_Start,
  clbr_phaseLow,
  clbr_phaseHigh,
} calibration_t;

typedef enum {
  s_start=0,
  s_measShunt,
  s_measOut,
  s_sleep,
  s_turnOnOut,
  s_waitOutStab,
  s_display
} step_t;

typedef struct {
  unsigned no_correction_adc:1;
  unsigned uart_en:1;
} eeprom_flags_t;

typedef struct {
  calibration_t clbr_mode:2;
  unsigned ds18exists:1;
  range_mode_t range_mode:2;
} operate_flags_t;

typedef struct {
  eeprom_flags_t flags;
  uint16_t shunt_nominal0; // номинал для больших проводимостей
  uint16_t shunt_nominal1; // номинал для малых проводимостей
  uint16_t sleep_time;  // msec
  uint16_t range_levelBottom;
  uint16_t range_levelTop   ;
  float    outDivider;
  float    conductivity_multi;
  float    analog_calc_k;
} settings_t;

typedef struct {
  unsigned long   summ;
  uint16_t    count;
} measure_t;

typedef struct {
  unsigned long conductivity;
  uint8_t spirit_prc;
} spirit_t;

#define WAIT_END_CONVERSATION while(ADCSRA & _BV(ADSC))
//установка мультиплексора ADC на заданный канал
#define SET_ADC_CHANNEL(C)  do {WAIT_END_CONVERSATION; ADMUX = _BV(REFS1) | _BV(REFS0) | (C);} while (0)
//запускает непрерывный режим измерения ADC
#define ADC_FREE_RUNNING (ADCSRA |=  (_BV(ADIE)|_BV(ADATE)|_BV(ADSC)))
//выключает непрерывный режим измерений ADC
#define ADC_STOP_FREE_RUNNING (ADCSRA &= (~( _BV(ADIE) | _BV(ADATE) )))
// запуск одиночного преобразования ADC
#define ADC_START_CONVERSATION do {WAIT_END_CONVERSATION; ADCSRA |= _BV(ADSC);}while (0)

#ifdef DISPLAY_ENABLE
 #if DISPLAY_MODEL == LCD1602
  #include <LiquidCrystal.h>
  LiquidCrystal lcd(RS_PIN, E_PIN, D4_PIN, D5_PIN, D6_PIN, D7_PIN);
 #elif DISPLAY_MODEL == LCD1602_I2C_PCF8574
  #include <LiquidCrystal_I2C.h>
  //==!!!========= укажи I2C адрес своего дисплея!=============
  LiquidCrystal_I2C lcd(0x27,16,2);  // set the LCD address to 0x27 for a 16 chars and 2 line display
 #endif
#endif

//==================================================================
void   calibration(long ref_resistor);
void   change_range(void);
void   checkButton(void);
uint16_t correctedOutADC(void);
void   defaultSettings(void);
long   get_conductivity(long resistance);
void   init_pins();
void   initADC();
void   initTimers();
void   out2display();
void   readSerial();
void   readSettings(void);
long   sensor_resistance(void);
void   serial_out(long cnd);
void   setAnalogOut(unsigned long c);
void   start_measure(uint8_t adc_ch,uint16_t measures_count);
void   writeSettings(void);

//===================Глобальные переменные==================
#ifndef REDUCE_SIZE    
  OneWire oneWire(ONE_WIRE_BUS);
  DallasTemperature sensors(&oneWire);
  DeviceAddress ds18b20_addr;
  float  sensorT;
#endif

volatile measure_t measures;
volatile uint16_t adc_shunt_avg;
uint16_t avg_out;
settings_t  cfg;
operate_flags_t oper_flags;
volatile long  r;

/*
 * ********************************************************
 */
void setup(void) {
  init_pins();
  Serial.begin(9600);
  cli();//запретим прерывания на время конфигурирования
  initADC();
  initTimers();
  sei();//разрешаем прерывания
#ifdef DISPLAY_ENABLE
  #if DISPLAY_MODEL == LCD1602_I2C_PCF8574
    lcd.setBacklight(true);
    lcd.init();
  #elif DISPLAY_MODEL == LCD1602
    lcd.begin(16, 2);
  #endif
#endif

    INT_OFF;
    readSettings();
#ifndef REDUCE_SIZE    
    for (int i=0;i<4; i++) {
      sensors.begin();
      if (sensors.getDeviceCount()) break;
      delay(100);
    }
    if (sensors.getDeviceCount()){
      if (sensors.getAddress(ds18b20_addr,0)) {
        sensors.setWaitForConversion(false);
        oper_flags.ds18exists = 1;
      }
    }
#endif    
    INT_ON;
}

//==================================================================================
void loop(void) {
  #ifndef REDUCE_SIZE    
  static  unsigned long dsCheckTime;
  #endif
  static  unsigned long wakeup_time;
  static    step_t  step;

  switch (step) {
  case  s_start:
  case  s_turnOnOut:
    if (! IS_OUTPUT_ON){
      OUTPUT_ON;
      wakeup_time = millis()+OUT_WAKEUP_TIME_MS;
    }
    step = s_waitOutStab;
    break;
  case  s_waitOutStab:
    if (wakeup_time<millis()){
      start_measure(ADC_OUT,OUT_ADC_MEASURES);
      step = s_measOut;
    }
    break;
  case  s_measOut:
    if (MEASURE_FINISHED){
      avg_out= CALC_AVG_ADC(OUT_ADC_MEASURES); //снимаем замер накачки
      start_measure(ADC_SHUNT,SHUNT_ADC_MEASURES);//стартуем замер на шунте
      step = s_measShunt;
    }
    break;
  case  s_measShunt:
    if (!MEASURE_FINISHED)  break;
    adc_shunt_avg= CALC_AVG_ADC(SHUNT_ADC_MEASURES); // снимаем замер шунта

    if (cfg.sleep_time>OUT_WAKEUP_TIME_MS){
      OUTPUT_OFF;//выкл накачку
      wakeup_time = millis()+cfg.sleep_time-OUT_WAKEUP_TIME_MS;
    }
    r=sensor_resistance(); //сопротивление ДП в Ом
    setAnalogOut(get_conductivity(r));
    if (oper_flags.clbr_mode!=clbr_NO){
      calibration(0);
    }
    step = s_display;    /* no break */

  case  s_display:
#ifndef REDUCE_SIZE    
    if (oper_flags.ds18exists){ // есть датчик T
      INT_OFF;
      if (millis()>dsCheckTime){ //если пришло время снять показания
        sensorT = sensors.getTempC(ds18b20_addr);
        sensors.requestTemperatures(); //запускаем новое
        dsCheckTime = millis()+900;
      }
      INT_ON;
    }
#endif    
    change_range();
  #ifdef DISPLAY_ENABLE
    //вывод на дисплей
    out2display();
  #endif
    step = s_sleep;
    /* no break */
  case  s_sleep:
    if (wakeup_time<millis()){
      step = s_turnOnOut;
    }
    break;
  }
  //==========проверка последовательного порта
  if (Serial.available())  readSerial();
  //==========вывод показаний в uart
  if (cfg.flags.uart_en){
    serial_out(get_conductivity(r));
  }
  checkButton();
}

uint8_t getSerialNumber(long* value){
  uint8_t index = 0; // Index into array; where to store the character
  char inData[10]; // Allocate some space for the string
  char inChar=-1; // Where to store the character read
  do {
    for (int i=0;i<100;i++) {
      delay(5);
      if (Serial.available()) break;
    }
    if (Serial.available()) {
      if(index < 9){
        inChar = Serial.read();
        inData[index] = inChar; // Store it
        index++; // Increment where to write next
        inData[index] = 0; // Null terminate the string
      }
      else
        break;
    }
    else
      break;
  } while(1);

  if (index >1) {
    *value = atol(inData);
  }
  return index;
}

void readSerial(){
  long serial_value;
  char inChar=-1; // Where to store the character read
  inChar = Serial.read();
  switch (inChar){
#ifndef REDUCE_SIZE    
    case '?'://получить список команд
        Serial.println("==list of commands==:");
        Serial.println("'d'-CFG dump");
        Serial.println("'kXXXXX'-start calibration, XXXXX - value of Rref in Ohm, must be 50000..150000");
        Serial.println("'mX'- range switch mode,X may be 0(auto), 1(low range), 2(hi range)");
        Serial.println("'lXXXX'- set low range Rshunt,XXXXX - Ohm");
        Serial.println("'LXXXX'- set hi range Rshunt, XXXXX - Ohm");
        Serial.println("'sXXXX'- pause between measures,ms");
        Serial.println("'rXXX'-  set autorange bottom limit, XXX - adc value");
        Serial.println("'RXXX'-  set autorange top limit, XXX - adc value");
        Serial.println("'oXXXX'- set k correction for output meashure, XXXX - value*1000");
        Serial.println("'cXXXX'- set k correction for conductivity, XXXX - value*1000");
        Serial.println("'fXXXX'- set constant of formula calculated analog output voltage, XXXX - value*1000");
        Serial.println("'u'- OFF output to UART");
        Serial.println("'U'- ON output to UART");
        break;
    case 'd'://дамп конфига
        Serial.println("=====CFG dump:");
        Serial.print("conductivity multiplicator:");Serial.println(cfg.conductivity_multi);
        Serial.print("formula Uout coeff:");Serial.println(cfg.analog_calc_k);
        Serial.print("shunt0:");Serial.println(cfg.shunt_nominal0);
        Serial.print("shunt1:");Serial.println(cfg.shunt_nominal1);
        Serial.print("sleep time,msec:");Serial.println(cfg.sleep_time);
        Serial.print("outDivider:");Serial.println(cfg.outDivider);
        Serial.print("rangeSwitch Low Limit:");Serial.println(cfg.range_levelBottom);
        Serial.print("rangeSwitch Hi Limit:");Serial.println(cfg.range_levelTop);
        Serial.print("uart output enable:");Serial.println(cfg.flags.uart_en);
        Serial.print("ds18b20 sensor exists:");Serial.println(oper_flags.ds18exists);
        Serial.print("range_mode:");Serial.println(oper_flags.range_mode);
        Serial.println("===end CFG");
        while (Serial.available()) {inChar = Serial.read();}
        break;
#endif        
    case 'k'://калибровка. Формат 'cХХХХХ'\n, ХХХХХ - номинал калибровочного резистора в Ом,(30..50кОм): 'k47000'<CR>
      if (getSerialNumber(&serial_value)){
        if ((serial_value<30000)||(serial_value>300000)) {
          Serial.println("Error! Rc should be in the range 30000..300000.Value:");Serial.println(serial_value);
        }
        else {
          Serial.print("calibration for ");Serial.print(serial_value);Serial.println("Ohm started");
          oper_flags.clbr_mode = clbr_Start;
          calibration(serial_value);
        }
      }
      else {
        Serial.print("no ref R");Serial.println(inChar);
      }
      break;
    case 'm'://управление пределом измерения 0-авто , 1-нижний, 2-верхний
      if (getSerialNumber(&serial_value)){
        if ((serial_value>=0) && (serial_value<=2)){
          oper_flags.range_mode = (range_mode_t)serial_value;
          Serial.print("set range_mode:");Serial.println(serial_value);
        }
      }
      break;
    case 'l'://установка сопротивления шунта0, ом
      if (getSerialNumber(&serial_value)){
        if (serial_value){
          cfg.shunt_nominal0 = serial_value;
          writeSettings();
          Serial.print("set shunt nominal:");Serial.println(serial_value);
        }
      }
      break;
    case 'L'://установка сопротивления шунта1, ом
      if (getSerialNumber(&serial_value)){
        if (serial_value){
          cfg.shunt_nominal1 = serial_value;
          writeSettings();
          Serial.print("set shunt nominal:");Serial.println(serial_value);
        }
      }
      break;
    case 's'://установка времени паузы между измерениями, мсек
      if (getSerialNumber(&serial_value)){
        if (serial_value){
          cfg.sleep_time = serial_value;
          writeSettings();
          Serial.print("set sleep time,ms:");Serial.println(cfg.sleep_time);
        }
      }
      break;
    case 'r'://установка нижнего порога переключения пределов
      if (getSerialNumber(&serial_value)){
        if (serial_value){
          cfg.range_levelBottom = serial_value;
          writeSettings();
          Serial.print("set bottomLimit:");Serial.println(cfg.range_levelBottom);
        }
      }
      break;
    case 'R'://установка верхнего порога переключения пределов
      if (getSerialNumber(&serial_value)){
        if (serial_value){
          cfg.range_levelTop = serial_value;
          writeSettings();
          Serial.print("set topLimit:");Serial.println(cfg.range_levelTop);
        }
      }
      break;
    case 'o'://установка коэф.делителя накачки, в тысячных o1512 - установить коэф. 1.512
      if (getSerialNumber(&serial_value)){
        if (serial_value){
          cfg.outDivider= serial_value/1000.0;
          writeSettings();
          Serial.print("set o:");Serial.println(cfg.outDivider,4);
        }
      }
      break;
    case 'c'://установка коэф. корректировки показаний проводимости, в тысячных c1000 - установить коэф. 1.0
      if (getSerialNumber(&serial_value)){
        if (serial_value){
          cfg.conductivity_multi= serial_value/1000.0;
          writeSettings();
          Serial.print("set c:");Serial.println(cfg.conductivity_multi,4);
        }
      }
      break;
    case 'f'://установка коэф. формулы расчета U аналогового выхода, в тысячных f003 - установить коэф. 0.003
      if (getSerialNumber(&serial_value)){
        if (serial_value){
          cfg.analog_calc_k= serial_value/1000.0;
          writeSettings();
          Serial.print("set calc k:");Serial.println(cfg.analog_calc_k,4);
        }
      }
      break;
    case 'u'://выкл. выдачу в UART данных
      cfg.flags.uart_en = 0;
      break;
    case 'U'://выкл. выдачу в UART данных
      cfg.flags.uart_en = 1;
      break;
    default:
      Serial.print("ignored:");Serial.println(inChar);
  }
}

void init_pins(){
  //====================конфигурирование линий ввода/вывода============
  DDRB  |= _BV(PIN_OUT) | _BV(PIN_ANOUT);   //выход накачки, выход PWM
  PORTB |= _BV(PIN_BUTTON_LIMIT) | _BV(PIN_BUTTON_CLBR); // подтяжка на кнопки
}

void initADC(){
  //====================конфигурируем ADC
  DIDR0 = 0x3F & ~(_BV(ADC_SHUNT)|_BV(ADC_OUT)); //запрещаем цифровой вход для используемых пинов ADC
  ACSR  = _BV(ACD);           // Запрет аналогового компаратора
  ADMUX = _BV(REFS1) | _BV(REFS0) |   // опорное напряжение ADC - внутренний источник 1.1В
       ADC_SHUNT;         // и установим мультиплексор на канал измерения тока
  //== АЦП ===============
  //  ADC Prescaler задаем 128 (ADPS2:0=111)
#define ADC_DIV 128L
  //  При ядровой 16 МГц тактовая АЦП будет (16МГц/128)=125 кГц
  //  ADC меряет 13 тактов (НО:первое после включения - 25 тактов!)
#define ADC_MEAS_TICKS  13
#define ADC_MEASURE_GAP 2
  //  Итого, время одного измерения = (13/125кГц) = 0.104 мс
  ADCSRA =_BV(ADEN) |  // АЦП -  вкл
      _BV(ADIE)  | // прерывание по завершению измерения - вкл
       _BV(ADPS2) |
       _BV(ADPS1) |
       _BV(ADPS0);
  ADCSRB = 0;
}

void initTimers(){
  // ============таймер2 - обеспечивает
  // 1)выдачу сигнала накачки на пин OC2A (D11 PB3)
  // 2)запуск ADC с задержкой относительно фронта накачки, в прерывании COMB
  //--------------------------------------------------
  // Mode: CTC (010) - сброс по OCRA , WGM2:0=010
  // пин OC2A(D11 PB3)-режим "toggle" COM2A1:COM2A0=01;  пин OC2B (D3 PD3) отключен COM2B1:COM2B0=00
  TCCR2A = _BV(COM2A0) | _BV(WGM21);
  TCCR2B = _BV(CS22)|_BV(CS20); // 1 0 1 (prescaler 128) timer tick= 8us/freq=125kHz
  // частота накачки задана в OUTPUT_FREQ
  OCR2A = SEMIPERIOD_TICKS; //кол-во тиков полупериода частоты накачки
  OCR2B = SEMIPERIOD_TICKS - ADC_START_DELAY_TICKS;// кол-во тиков от фронта накачки до старта преобразования ADC
  TIMSK2 = _BV(OCIE2B); // разрешаем прерывания по компаратору B
  // ============таймер1 - обеспечивает
  // 1)выдачу ШИМ сигналов пропорциональных показаниям датчиков на пины OC1A (D9 PB1) OC1B(D10 PB2)
  // Mode 10 Phase Correct PWM(TOP=ICR1), 1 0 1 0
  // пин OC1A: COM1A1:COM1A0  10 - Clear OC1A when upcounting.Set OC1A on Compare Match when downcounting.
  TCCR1A = _BV(COM1A1) | _BV(WGM11);
  TCCR1B = _BV(WGM13) | _BV(CS10);
  ICR1 = TOP_TIMER1;
  OCR1A = 0;
}

//------------старт ADC (таймер2 равен OCR2B)
ISR (TIMER2_COMPB_vect){
  if ((PINB & _BV(PIN_OUT))) {        // при активном выходе
      if (measures.count)   // если кол-во измерений для усреднения не достигли
    {
      ADC_START_CONVERSATION;     //запускаем ADC
    }
  }
}

//------------прерывание по завершению измерения АЦП
ISR(ADC_vect){
  uint16_t ret;
  asm volatile(
    "lds %A0,  %1\n\t"
    "lds %B0,  %2"
    : "=w" (ret): "n"(&ADCL), "n"(&ADCH));
  measures.summ += ret;
  if (measures.count) measures.count --;
}

void serial_out(long cnd){
  static unsigned long time4UART=0;
  if (time4UART<millis()){
    Serial.println(cnd);
    time4UART = millis()+PERIOD_SERIAL_MS;
  }
}

/*
 * вернет сопротивление датчика, Ом
 */
long sensor_resistance(void){
  if (adc_shunt_avg==0) return 9999999.0;
  uint16_t u1=correctedOutADC();
  uint16_t u2=adc_shunt_avg;
  uint16_t shunt_nominal=IS_RANGE0? cfg.shunt_nominal0:cfg.shunt_nominal1;
  return max(((long)(((u1*1.0)/u2 - 1)*shunt_nominal)-RPROTECT_NOMINAL_OHM),0);
}

void   start_measure(uint8_t adc_ch,uint16_t measures_count){
  SET_ADC_CHANNEL(adc_ch);
  measures.count = 1; while (measures.count); // холостое измерение
  measures.summ = 0;
  measures.count = measures_count;
}


void set_range(range_mode_t range){
  switch (range){
  case range_mode_0R:
    SET_RANGE0;
    break;
  case range_mode_1R:
    SET_RANGE1;
    break;
  case range_mode_Auto:
    break;
  }
}

void change_range(void){
  // переключение пределов
  if (oper_flags.range_mode == range_mode_Auto) {
    if (IS_RANGE0){
      if (adc_shunt_avg < cfg.range_levelBottom) SET_RANGE1;
    }
    else {
      if (adc_shunt_avg > cfg.range_levelTop) SET_RANGE0;
    }
    return;
  }
  if (oper_flags.range_mode == range_mode_0R)
    SET_RANGE0;
  else
    SET_RANGE1;
}

uint16_t correctedOutADC(void){
  return (uint16_t)round(avg_out * cfg.outDivider)/2;
}

void   readSettings(void){
  if (eeprom_read_byte(EEPROM_ADR_VERSION)==EEPROM_VERSION){
    DBG("setting read ver");DBGLN(EEPROM_VERSION);
    eeprom_read_block(&cfg,EEPROM_ADR_CFG,sizeof(cfg));
  }
  else {
    DBGLN("def setting write");
    defaultSettings();
    writeSettings();
  }
}

void   writeSettings(void){
  while (!eeprom_is_ready());
  if (eeprom_read_byte(EEPROM_ADR_VERSION)!=EEPROM_VERSION){
    eeprom_write_byte(EEPROM_ADR_VERSION,EEPROM_VERSION);
  }
  while (!eeprom_is_ready());
  eeprom_write_block(&cfg,EEPROM_ADR_CFG,sizeof(cfg));
  while (!eeprom_is_ready());
}

void   defaultSettings(void){
  cfg.flags.no_correction_adc = 0;
  cfg.shunt_nominal0 = SHUNT0_NOMINAL_OHM;
  cfg.shunt_nominal1 = SHUNT1_NOMINAL_OHM;
  cfg.sleep_time = SLEEP_PAUSE_MS;
  cfg.outDivider = OUT_DIVIDER;
  cfg.range_levelBottom = LOW_LIMIT_RANGE;
  cfg.range_levelTop    = TOP_LIMIT_RANGE;
  cfg.conductivity_multi = CONDUCTIVITY_MULTI;
  cfg.analog_calc_k = AN_OUT_CALCULATE_COEFF;
}

long getShuntNominal(long ref_resistor){
  DBG("SH");DBG(":");DBG(adc_shunt_avg);DBG(" o:");DBG(avg_out);
  double f = (ref_resistor + RPROTECT_NOMINAL_OHM)/((double)correctedOutADC()/adc_shunt_avg - 1.0);
  DBG(" calc.nominal:");DBGLN(f);
  return round(f);
}

void calibration(long ref_resistor){
static  long ref;
static  unsigned long stoptime;
  if (oper_flags.clbr_mode == clbr_NO) return;
  switch (oper_flags.clbr_mode){
  case clbr_NO:
    break;
  case clbr_Start:
    ref = ref_resistor;
    oper_flags.clbr_mode=clbr_phaseLow;
    oper_flags.range_mode = range_mode_0R;
    stoptime = millis()+2000;
    break;
  case clbr_phaseLow:
    if (millis()<stoptime) return;
    cfg.shunt_nominal0 = getShuntNominal(ref);
    cfg.range_levelBottom = adc_shunt_avg-2;
    Serial.print("low range calibration DONE: R=");Serial.println(cfg.shunt_nominal0);
    oper_flags.clbr_mode=clbr_phaseHigh;
    oper_flags.range_mode = range_mode_1R;
    stoptime = millis()+2000;
    break;
  case clbr_phaseHigh:
    if (millis()<stoptime) return;
    cfg.shunt_nominal1 = getShuntNominal(ref);
    cfg.range_levelTop = adc_shunt_avg;
    Serial.print("hi range calibration DONE: R2=");Serial.println(cfg.shunt_nominal1);
    writeSettings();
    oper_flags.clbr_mode=clbr_NO;
    oper_flags.range_mode = range_mode_Auto;
    Serial.println("calibration done");
    stoptime = 0;
    break;
  }
}

void   checkButton(void){
static unsigned long next_time;
static struct {
  unsigned plimit:1;
  unsigned pclbr:1;;
} prev_pressed;
  if (millis()<next_time)  return;
  next_time = millis() + 300;

  if (!(PINB & _BV(PIN_BUTTON_LIMIT))){
  if (prev_pressed.plimit==0){
    prev_pressed.plimit=1;
    oper_flags.range_mode++;
    if (oper_flags.range_mode>range_mode_1R)
      oper_flags.range_mode = range_mode_Auto;
  }
  }
  else
    prev_pressed.plimit=0;

  if (!(PINB & _BV(PIN_BUTTON_CLBR))){
  if (!prev_pressed.pclbr){
    if (oper_flags.clbr_mode == clbr_NO){
    oper_flags.clbr_mode = clbr_Start;
    calibration(CALIBRATION_RESISTOR);
    }
    prev_pressed.pclbr=1;
  }
  }
  else
    prev_pressed.pclbr=0;
}

void setAnalogOut(unsigned long c){
  if (!c) OCR1A=0;
  //f(x)=(1 - 1/(COEFF*x+1))
  double tmp= 1.0 - 1.0/(cfg.analog_calc_k*c +1);
  OCR1A = min((uint16_t)TOP_TIMER1*(tmp*MAX_PRC_AOUT)/100,TOP_TIMER1);
}

long get_conductivity(long resistance){
  if (!resistance) return MAX_CONDUCTIVITY;
  if (adc_shunt_avg>0) return round(10000000.0*cfg.conductivity_multi/(resistance)); //десятые доли мксименс
  return 0;
}

void out2display(){
  char str[30];
  if (oper_flags.clbr_mode==clbr_NO){
    sprintf (str, "%s:%-8ld u:%-3d", (oper_flags.range_mode == range_mode_Auto?"A":(IS_RANGE0?"f":"F")), get_conductivity(r),adc_shunt_avg);
  }
  else
    sprintf (str, "CLBR %d    u:%-3d",oper_flags.clbr_mode, adc_shunt_avg);
  lcd.setCursor(0, 0);lcd.print(&str[0]);

  lcd.setCursor(0, 1); lcd.print("R:");
  if (adc_shunt_avg){
        if (r>1000000l){
          lcd.print((float)r/1000000.0,2);lcd.print("M      "); // сопротивление
        }
        else {
          lcd.print((float)r/1000,0);lcd.print("k      "); // сопротивление
        }
  }
  else {
    lcd.print("----     ");// ХХ
  }

#ifndef REDUCE_SIZE    
  if (oper_flags.ds18exists){
    lcd.setCursor(10, 1); lcd.print("T:");lcd.print(sensorT,1);// температура
  }
  else 
#endif  
  {
    lcd.setCursor(11, 1);lcd.print("o:"); lcd.print(round(correctedOutADC()*1.1));
  }
}
