"""
Менеджер конфигурации приложения
"""
import configparser
import os

class ConfigManager:
    """Управление настройками приложения"""
    
    def __init__(self, app_name="COM_Terminal"):
        self.app_name = app_name
        self.config_file = "config.ini"
        self.config = configparser.ConfigParser()
        
        # Настройки по умолчанию
        self.defaults = {
            'Window': {
                'width': '1000',
                'height': '700',
                'x': '100',
                'y': '100',
                'maximized': 'false'
            },
            'Terminal': {
                'bg_color': '#0a1929',
                'text_color': '#e6f7ff',
                'font_family': 'Consolas',
                'font_size': '9'
            },
            'HC12': {
                'channel': '001',
                'baud': '9600',
                'power': 'P8 (+20 dBm)',
                'mode': 'FU1'
            },
            'General': {
                'show_timestamp': 'true'
            }
        }
        
        # Загружаем конфиг
        self.load_config()
    
    def load_config(self):
        """Загрузить конфигурацию из файла"""
        if os.path.exists(self.config_file):
            self.config.read(self.config_file, encoding='utf-8')
        else:
            # Создаем конфиг с настройками по умолчанию
            for section, options in self.defaults.items():
                if not self.config.has_section(section):
                    self.config.add_section(section)
                for key, value in options.items():
                    self.config.set(section, key, value)
            self.save_config()
    
    def save_config(self):
        """Сохранить конфигурацию в файл"""
        with open(self.config_file, 'w', encoding='utf-8') as f:
            self.config.write(f)
    
    def get_window_settings(self):
        """Получить настройки окна"""
        section = 'Window'
        if not self.config.has_section(section):
            return self.defaults[section]
        
        return {
            'width': int(self.config.get(section, 'width', fallback=self.defaults[section]['width'])),
            'height': int(self.config.get(section, 'height', fallback=self.defaults[section]['height'])),
            'x': int(self.config.get(section, 'x', fallback=self.defaults[section]['x'])),
            'y': int(self.config.get(section, 'y', fallback=self.defaults[section]['y'])),
            'maximized': self.config.getboolean(section, 'maximized', fallback=False)
        }
    
    def save_window_settings(self, window):
        """Сохранить настройки окна"""
        if not window.isMaximized():
            geometry = window.geometry()
            self.config.set('Window', 'width', str(geometry.width()))
            self.config.set('Window', 'height', str(geometry.height()))
            self.config.set('Window', 'x', str(geometry.x()))
            self.config.set('Window', 'y', str(geometry.y()))
        self.config.set('Window', 'maximized', str(window.isMaximized()))
        self.save_config()
    
    def get_terminal_colors(self):
        """Получить цвета терминала"""
        section = 'Terminal'
        if not self.config.has_section(section):
            return self.defaults[section]
        
        return {
            'bg_color': self.config.get(section, 'bg_color', fallback=self.defaults[section]['bg_color']),
            'text_color': self.config.get(section, 'text_color', fallback=self.defaults[section]['text_color']),
            'font_family': self.config.get(section, 'font_family', fallback=self.defaults[section]['font_family']),
            'font_size': self.config.get(section, 'font_size', fallback=self.defaults[section]['font_size'])
        }
    
    def save_terminal_colors(self, bg_color, text_color, font_family, font_size):
        """Сохранить цвета терминала"""
        self.config.set('Terminal', 'bg_color', bg_color)
        self.config.set('Terminal', 'text_color', text_color)
        self.config.set('Terminal', 'font_family', font_family)
        self.config.set('Terminal', 'font_size', str(font_size))
        self.save_config()
    
    def get_hc12_settings(self):
        """Получить настройки HC-12"""
        section = 'HC12'
        if not self.config.has_section(section):
            return self.defaults[section]
        
        return {
            'channel': self.config.get(section, 'channel', fallback=self.defaults[section]['channel']),
            'baud': self.config.get(section, 'baud', fallback=self.defaults[section]['baud']),
            'power': self.config.get(section, 'power', fallback=self.defaults[section]['power']),
            'mode': self.config.get(section, 'mode', fallback=self.defaults[section]['mode'])
        }
    
    def save_hc12_settings(self, channel, baud, power, mode):
        """Сохранить настройки HC-12"""
        self.config.set('HC12', 'channel', channel)
        self.config.set('HC12', 'baud', baud)
        self.config.set('HC12', 'power', power)
        self.config.set('HC12', 'mode', mode)
        self.save_config()
        
    def get_timestamp_setting(self):
        """Получить настройку Timestamp"""
        section = 'General'
        if not self.config.has_section(section):
            return True
        return self.config.getboolean(section, 'show_timestamp', fallback=True)
    
    def save_timestamp_setting(self, show_timestamp):
        """Сохранить настройку Timestamp"""
        if not self.config.has_section('General'):
            self.config.add_section('General')
        self.config.set('General', 'show_timestamp', str(show_timestamp))
        self.save_config()