"""
Базовый класс эмулятора
"""
import time
from datetime import datetime
import re
import random
import socket
import sys

class BaseEmulator:
    """Базовый класс для всех эмуляторов"""
    def __init__(self):
        self.enabled = False
        self.device_type = "BASE"
        self.parent = None  # Ссылка на главное окно для доступа к интерфейсу
        
        # Общие параметры для всех эмуляторов
        self.triac_shorted = False  # Режим "пробит семистор"
        self.max_power = 3000  # Максимальная мощность для ТЭНа (Ватт)
        self.max_voltage = 250  # Максимальное напряжение (Вольт)
        
        # Флаг для определения типа устройства
        self.is_power_device = False  # True для устройств с мощностью, False для устройств с напряжением
        
    def set_parent(self, parent):
        """Установить ссылку на родительское окно"""
        self.parent = parent
        
    def set_enabled(self, enabled: bool):
        """Включить/выключить эмулятор"""
        self.enabled = enabled
        print(f"[{self.device_type}] Эмулятор {'включен' if enabled else 'выключен'}")
    
    def process_command(self, command: str):
        """Обработка команды (должен быть переопределен)"""
        return None
    
    def periodic_update(self):
        """Периодическое обновление (для эмуляторов с авто-отправкой данных)"""
        pass
    
    def start(self):
        """Запуск эмулятора (для UDP и других активных эмуляторов)"""
        pass
    
    def stop(self):
        """Остановка эмулятора"""
        pass
    
    def apply_triac_effect(self):
        """Применить эффект пробитого семистора (должен быть переопределен в дочерних классах)"""
        # Базовая реализация - ничего не делает
        # В дочерних классах нужно переопределить для конкретного типа устройства
        pass
    
    def set_triac_shorted(self, shorted: bool):
        """Установить режим "пробит семистор" """
        old_state = self.triac_shorted
        self.triac_shorted = shorted
        
        if old_state != shorted:
            print(f"[{self.device_type}] Режим 'пробит семистор': {'ВКЛ' if shorted else 'ВЫКЛ'}")
            self.apply_triac_effect()
        
        return self.triac_shorted