"""
Эмулятор КВИК (KVIC)
"""
import time
import random
from emulators.base import BaseEmulator
from utils import get_timestamp

class KVICEmulator(BaseEmulator):
    """Эмулятор КВИК (KVIC)"""
    def __init__(self):
        super().__init__()
        self.device_type = "KVIC"
        self.current_voltage = 0.0  # Текущее напряжение (в децивольтах)
        self.target_voltage = 0.0   # Целевое напряжение
        self.mode = "2"             # Режим: "0"-рабочий, "1"-разгон, "2"-сон
        self.power_on = False
        self.last_report_time = 0
        self.report_interval = 200  # мс (частая отправка отчетов как в реальном устройстве)
        
        # Для этого устройства
        self.is_power_device = False  # Работает с напряжением
        self.normal_max_voltage = 230.0  # Нормальное максимальное напряжение
        self.triac_max_voltage = 250.0   # Максимальное напряжение при пробитом семисторе
    
    def apply_triac_effect(self):
        """Применить эффект пробитого семистора для КВИК"""
        if self.triac_shorted and self.power_on:
            # Если семистор пробит и устройство включено, напряжение максимальное
            self.current_voltage = self.triac_max_voltage
            self.mode = "0"  # Принудительно в рабочий режим
            print(f"[{self.device_type}] Применен эффект пробитого семистора: напряжение {self.current_voltage}В")
        elif not self.triac_shorted and self.power_on:
            # Если семистор исправен, восстанавливаем нормальные значения
            if self.target_voltage > self.normal_max_voltage:
                self.target_voltage = self.normal_max_voltage
            if self.current_voltage > self.normal_max_voltage:
                self.current_voltage = min(self.target_voltage, self.normal_max_voltage)
    
    def periodic_update(self):
        """Автоматическая отправка отчетов"""
        if not self.enabled:
            return
        
        current_time = time.time() * 1000
        if current_time - self.last_report_time >= self.report_interval:
            self.send_report()
            self.last_report_time = current_time
    
    def send_report(self):
        """Отправить отчет в формате T + 6 hex символов"""
        # Формируем ответ: T + 6 hex символов
        # Формат: T[текущее][целевое][режим]
        current_hex = f"{int(self.current_voltage * 10):03X}"  # В децивольтах
        target_hex = f"{int(self.target_voltage * 10):03X}"
        mode_hex = self.mode
        report = f"T{current_hex}{target_hex}{mode_hex}"
        
        if self.parent and hasattr(self.parent, 'serial_manager'):
            try:
                # Используем serial_manager для отправки
                if self.parent.serial_manager.is_connected():
                    self.parent.serial_manager.send_text(report + "\r")
                    if self.parent:
                        timestamp = get_timestamp(self.parent)
                        self.parent.terminal.append(f"{timestamp}[Авто-отчет] KVIC >> {report}")
            except Exception as e:
                print(f"[KVIC] Ошибка отправки авто-отчета: {e}")
    
    def process_command(self, command: str):
        """Обработка команды КВИК"""
        if not self.enabled:
            return None
        
        cmd = command.strip()
        print(f"[KVIC] Обработка команды: {cmd}")
        
        # Проверяем эффект пробитого семистора
        if self.triac_shorted and self.power_on:
            print(f"[KVIC] Внимание: семистор пробит, некоторые команды могут быть ограничены")
        
        # Команды КВИК:
        # Txxxxxx - запрос состояния (ответ: T+6 hex символов)
        if cmd.startswith("T"):
            # Формируем ответ: T + 6 hex символов
            current_hex = f"{int(self.current_voltage * 10):03X}"
            target_hex = f"{int(self.target_voltage * 10):03X}"
            mode_hex = self.mode
            response = f"T{current_hex}{target_hex}{mode_hex}"
            return response
        
        # Sxxx - установка напряжения (в децивольтах)
        elif cmd.startswith("S"):
            try:
                voltage_decivolts = int(cmd[1:], 16)  # Hex to int
                voltage = voltage_decivolts / 10.0
                
                # Если семистор пробит, ограничиваем максимальное напряжение
                if self.triac_shorted:
                    max_voltage = self.triac_max_voltage
                else:
                    max_voltage = self.normal_max_voltage
                
                if voltage <= max_voltage:
                    self.target_voltage = voltage
                    if self.target_voltage > 0:
                        # Если семистор пробит, устанавливаем максимальное напряжение
                        if self.triac_shorted:
                            self.current_voltage = self.triac_max_voltage
                        else:
                            self.current_voltage = self.target_voltage
                        self.mode = "0"  # Работа
                        self.power_on = True
                    else:
                        self.current_voltage = 0
                        self.mode = "2"  # Сон
                        self.power_on = False
                    # KVIC не отвечает на команду установки напряжения
                    return None
                else:
                    return None  # Игнорируем команду если превышен лимит
            except:
                return None
        
        # Mx - установка режима
        elif cmd.startswith("M"):
            mode = cmd[1:]
            if mode in ["0", "1", "2", "3"]:
                self.mode = mode
                if mode == "2":  # Сон
                    self.power_on = False
                    self.current_voltage = 0
                    self.target_voltage = 0
                else:
                    self.power_on = True
                    if mode == "1":  # Разгон
                        # Если семистор пробит - максимальное напряжение
                        if self.triac_shorted:
                            self.current_voltage = self.triac_max_voltage
                            self.target_voltage = self.triac_max_voltage
                        else:
                            self.current_voltage = self.normal_max_voltage
                            self.target_voltage = self.normal_max_voltage
                # KVIC не отвечает на команду установки режима
                return None
            return None
        
        else:
            return None