"""
Менеджер эмуляторов
"""
from emulators.rmvk import RMVKEmulator
from emulators.kvic import KVICEmulator
from emulators.stab_avr import StabAVREmulator
from emulators.universal import UniversalProtocolEmulator
from emulators.modbus import ModbusRTUEmulator
from emulators.udp_stab import UDPStabEmulator

class EmulatorManager:
    """Менеджер для управления всеми эмуляторами"""
    def __init__(self, parent=None):
        self.emulators = {
            "RMVK": RMVKEmulator(),
            "KVIC": KVICEmulator(),
            "KVIC9600": KVICEmulator(),
            "STAB_AVR": StabAVREmulator(),
            "UNI_PROTOCOL": UniversalProtocolEmulator(),
            "MODBUS_RTU": ModbusRTUEmulator(),
            "UDP_STAB": UDPStabEmulator(),
        }
        self.current_emulator = "RMVK"  # По умолчанию
        self.parent = parent
        
        # Устанавливаем родителя для всех эмуляторов
        for emulator in self.emulators.values():
            emulator.set_parent(parent)
        
        # Настройка KVIC9600
        self.emulators["KVIC9600"].device_type = "KVIC9600"
    
    def set_current(self, emulator_name: str):
        """Установить активный эмулятор"""
        if emulator_name in self.emulators:
            # Останавливаем предыдущий эмулятор если он активный (например, UDP)
            old_emulator = self.get_current()
            if old_emulator and old_emulator.enabled and hasattr(old_emulator, 'stop'):
                old_emulator.stop()
            
            self.current_emulator = emulator_name
            
            # Запускаем новый если он включен
            new_emulator = self.get_current()
            if new_emulator and new_emulator.enabled and hasattr(new_emulator, 'start'):
                new_emulator.start()
            
            print(f"[EMU MANAGER] Активирован эмулятор: {emulator_name}")
            return True
        return False
    
    def get_current(self):
        """Получить текущий активный эмулятор"""
        return self.emulators.get(self.current_emulator)
    
    def enable_current(self, enabled: bool):
        """Включить/выключить текущий эмулятор"""
        emulator = self.get_current()
        if emulator:
            emulator.set_enabled(enabled)
            return True
        return False
    
    def process_command(self, command: str, is_binary=False):
        """Обработать команду текущим эмулятором"""
        emulator = self.get_current()
        if not emulator or not emulator.enabled:
            return None
        
        # Особый случай для Modbus (бинарные данные)
        if self.current_emulator == "MODBUS_RTU" and is_binary:
            return emulator.process_command(command)
        elif self.current_emulator == "MODBUS_RTU" and not is_binary:
            # Попробуем обработать как hex строку
            return emulator.process_command(command)
        else:
            # Текстовые команды
            return emulator.process_command(command)
    
    def periodic_update(self):
        """Периодическое обновление всех эмуляторов"""
        for emulator in self.emulators.values():
            if emulator.enabled and hasattr(emulator, 'periodic_update'):
                emulator.periodic_update()