"""
Эмулятор Modbus RTU устройства
"""
from emulators.base import BaseEmulator

class ModbusRTUEmulator(BaseEmulator):
    """Эмулятор Modbus RTU устройства"""
    def __init__(self):
        super().__init__()
        self.device_type = "MODBUS_RTU"
        self.address = 10
        self.registers = {
            0x2000: 0x0000,  # Control register
            0x2001: 0x0000,  # Frequency register (0-5000 = 0-50Hz)
            0x3000: 0x0000,  # Status register
            0x3001: 0x1388,  # Current frequency (5000 = 50Hz)
        }
        self.started = False
        
    def process_command(self, data_bytes):
        """Обработка Modbus RTU фрейма"""
        if not self.enabled:
            return None
        
        if isinstance(data_bytes, str):
            try:
                # Преобразуем hex строку в байты
                data_bytes = bytes.fromhex(data_bytes)
            except:
                return None
        
        if len(data_bytes) < 6:  # Минимальная длина Modbus RTU фрейма
            return None
        
        # Извлекаем адрес устройства
        address = data_bytes[0]
        if address != self.address:
            return None  # Не наш адрес
        
        # Извлекаем функцию
        function = data_bytes[1]
        
        response = None
        
        # Функция 0x03: Read Holding Registers
        if function == 0x03:
            start_addr = (data_bytes[2] << 8) | data_bytes[3]
            reg_count = (data_bytes[4] << 8) | data_bytes[5]
            
            # Формируем ответ
            response = bytearray()
            response.append(self.address)
            response.append(0x03)
            response.append(reg_count * 2)  # Byte count
            
            for i in range(reg_count):
                reg_addr = start_addr + i
                value = self.registers.get(reg_addr, 0x0000)
                response.append((value >> 8) & 0xFF)
                response.append(value & 0xFF)
        
        # Функция 0x06: Write Single Register
        elif function == 0x06:
            reg_addr = (data_bytes[2] << 8) | data_bytes[3]
            reg_value = (data_bytes[4] << 8) | data_bytes[5]
            
            # Записываем значение в регистр
            self.registers[reg_addr] = reg_value
            
            # Эхо-ответ
            response = data_bytes[:6]  # Возвращаем то же самое
        
        # Функция 0x10: Write Multiple Registers
        elif function == 0x10:
            start_addr = (data_bytes[2] << 8) | data_bytes[3]
            reg_count = (data_bytes[4] << 8) | data_bytes[5]
            byte_count = data_bytes[6]
            
            # Записываем регистры
            data_idx = 7
            for i in range(reg_count):
                reg_addr = start_addr + i
                if data_idx + 1 < len(data_bytes):
                    value = (data_bytes[data_idx] << 8) | data_bytes[data_idx + 1]
                    self.registers[reg_addr] = value
                    data_idx += 2
            
            # Ответ подтверждения
            response = bytearray()
            response.append(self.address)
            response.append(0x10)
            response.append(data_bytes[2])  # Start address high
            response.append(data_bytes[3])  # Start address low
            response.append(data_bytes[4])  # Register count high
            response.append(data_bytes[5])  # Register count low
        
        if response:
            # Добавляем CRC
            crc = self.calculate_crc(response)
            response.append(crc & 0xFF)
            response.append((crc >> 8) & 0xFF)
            
            print(f"[MODBUS] Ответ: {response.hex()}")
            return bytes(response)
        
        return None
    
    def calculate_crc(self, data):
        """Расчет CRC16 для Modbus"""
        crc = 0xFFFF
        for byte in data:
            crc ^= byte
            for _ in range(8):
                if crc & 0x0001:
                    crc = (crc >> 1) ^ 0xA001
                else:
                    crc = crc >> 1
        return crc