"""
Эмулятор РМВК
"""
import re
from emulators.base import BaseEmulator

class RMVKEmulator(BaseEmulator):
    """Эмулятор РМВК"""
    def __init__(self):
        super().__init__()
        self.device_type = "RMVK"
        self.voltage_in = 220
        self.voltage_out = 0
        self.voltage_set = 0
        self.power_on = False
        self.memory = [220, 200, 180, 160, 140, 120, 100, 80, 60, 40]
        self.mode = "2"  # 0-работа, 1-разгон, 2-сон, 3-авария
        
        # Для этого устройства (регулирует напряжение)
        self.is_power_device = False  # False для устройств с напряжением
        self.normal_max_voltage = 230  # Нормальное максимальное напряжение
        self.triac_max_voltage = 250   # Максимальное напряжение при пробитом семисторе
    
    def apply_triac_effect(self):
        """Применить эффект пробитого семистора для РМВК"""
        if self.triac_shorted and self.power_on:
            # Если семистор пробит и устройство включено, напряжение максимальное
            self.voltage_out = self.triac_max_voltage
            self.mode = "0"  # Принудительно в рабочий режим
            print(f"[{self.device_type}] Применен эффект пробитого семистора: напряжение {self.voltage_out}В")
        elif not self.triac_shorted and self.power_on:
            # Если семистор исправен, восстанавливаем нормальные значения
            if self.voltage_set > self.normal_max_voltage:
                self.voltage_set = self.normal_max_voltage
            if self.voltage_out > self.normal_max_voltage:
                self.voltage_out = min(self.voltage_set, self.normal_max_voltage)
    
    def process_command(self, command: str):
        """Обработка команды РМВК"""
        if not self.enabled:
            return None
        
        # Нормализация команды (русские буквы -> английские)
        cmd = command.replace('А', 'A').replace('Т', 'T').strip()
        
        print(f"[RMVK] Обработка команды: {cmd}")
        
        # Проверяем эффект пробитого семистора перед обработкой
        if self.triac_shorted and self.power_on:
            print(f"[RMVK] Внимание: семистор пробит, напряжение ограничено {self.triac_max_voltage}В")
        
        # AT+VI? - напряжение в сети
        if cmd == "AT+VI?":
            return f"{self.voltage_in:03d}"
        
        # AT+VO? - выходное напряжение
        elif cmd == "AT+VO?":
            if not self.power_on:
                return "000"
            return f"{self.voltage_out:03d}"
        
        # AT+VS? - установленное напряжение
        elif cmd == "AT+VS?":
            return f"{self.voltage_set:03d}"
        
        # AT+VS=xxx - установить напряжение
        elif cmd.startswith("AT+VS="):
            match = re.match(r"AT\+VS=(\d{2,3})", cmd)
            if match:
                voltage = int(match.group(1))
                
                # Если семистор пробит, ограничиваем максимальное напряжение
                if self.triac_shorted:
                    max_voltage = self.triac_max_voltage
                else:
                    max_voltage = self.normal_max_voltage
                
                if 40 <= voltage <= max_voltage:
                    self.voltage_set = voltage
                    if self.power_on and voltage > 0:
                        # Если семистор пробит, устанавливаем максимальное напряжение
                        if self.triac_shorted:
                            self.voltage_out = self.triac_max_voltage
                        else:
                            self.voltage_out = voltage
                        self.mode = "0"  # Работа
                    elif voltage == 0:
                        self.voltage_out = 0
                        self.mode = "2"  # Сон
                    return f"{voltage:03d}"
                else:
                    # Если команда пытается установить напряжение выше лимита
                    if self.triac_shorted and voltage > self.triac_max_voltage:
                        print(f"[RMVK] Семистор пробит, напряжение ограничено {self.triac_max_voltage}В")
                    return "error"
            return "error"
        
        # AT+ON? - состояние
        elif cmd == "AT+ON?":
            return "ON" if self.power_on else "OFF"
        
        # AT+ON=x - включить/выключить
        elif cmd.startswith("AT+ON="):
            match = re.match(r"AT\+ON=(\d)", cmd)
            if match:
                state = int(match.group(1))
                self.power_on = (state == 1)
                if self.power_on:
                    # Если семистор пробит, устанавливаем максимальное напряжение
                    if self.triac_shorted:
                        self.voltage_out = self.triac_max_voltage
                        self.voltage_set = self.triac_max_voltage
                    else:
                        self.voltage_out = self.voltage_set
                    self.mode = "0" if self.voltage_out > 0 else "2"
                else:
                    self.voltage_out = 0
                    self.mode = "2"  # Сон
                return "ON" if self.power_on else "OFF"
            return "error"
        
        # AT+SM=x - выбрать из памяти
        elif cmd.startswith("AT+SM="):
            match = re.match(r"AT\+SM=(\d)", cmd)
            if match:
                mem_idx = int(match.group(1))
                if 0 <= mem_idx <= 9:
                    voltage = self.memory[mem_idx]
                    
                    # Если семистор пробит, проверяем лимит
                    if self.triac_shorted and voltage > self.triac_max_voltage:
                        voltage = self.triac_max_voltage
                        self.memory[mem_idx] = voltage  # Обновляем значение в памяти
                    
                    self.voltage_set = voltage
                    if self.power_on and voltage > 0:
                        # Если семистор пробит, устанавливаем максимальное напряжение
                        if self.triac_shorted:
                            self.voltage_out = self.triac_max_voltage
                        else:
                            self.voltage_out = voltage
                        self.mode = "0"  # Работа
                    return "OK"
                else:
                    return "error"
            return "error"
        
        # AT+SS? - статус (расширенная команда)
        elif cmd == "AT+SS?":
            return self.mode  # Возвращаем цифру режима
        
        # Неизвестная команда
        else:
            return None  # РМВК не отвечает на неизвестные команды