"""
Эмулятор стабилизатора AVR
"""
import re
import time
from emulators.base import BaseEmulator

class StabAVREmulator(BaseEmulator):
    """Эмулятор стабилизатора AVR (работает с мощностью)"""
    def __init__(self):
        super().__init__()
        self.device_type = "STAB_AVR"
        self.current_power = 0      # Текущая мощность (Вт)
        self.target_power = 0       # Целевая мощность (Вт)
        self.mode = "2"            # 0-работа, 1-разгон, 2-сон, 3-авария
        self.power_on = False       # Питание устройства
        self.use_prefixes = True    # По умолчанию с префиксами
        self.mode_prefix = "M"
        self.current_prefix = "C" 
        self.target_prefix = "T"
        
        # Для этого устройства - это стабилизатор мощности
        self.is_power_device = True   # Это устройство работает с мощностью!
        self.normal_max_power = 3000  # Нормальная максимальная мощность (Вт)
        self.triac_max_power = 3000   # Максимальная мощность при пробитом семисторе
    
    def apply_triac_effect(self):
        """Применить эффект пробитого семистора для стабилизатора мощности"""
        if self.triac_shorted and self.power_on:
            # Если семистор пробит и устройство включено, мощность максимальная
            self.current_power = self.triac_max_power
            self.mode = "0"  # Принудительно переводим в рабочий режим
            print(f"[{self.device_type}] Применен эффект пробитого семистора: мощность {self.current_power}Вт")
        elif not self.triac_shorted and self.power_on:
            # Если семистор исправен, восстанавливаем нормальные значения
            if self.target_power > self.normal_max_power:
                self.target_power = self.normal_max_power
            if self.current_power > self.normal_max_power:
                self.current_power = min(self.target_power, self.normal_max_power)
    
    def process_command(self, command: str):
        """Обработка команд STAB_AVR"""
        if not self.enabled:
            return None
        
        # Нормализация команды
        cmd = command.replace('А', 'A').replace('Т', 'T').strip()
        print(f"[AVR] Обработка команды: {cmd}")
        
        # Проверяем эффект пробитого семистора перед обработкой
        if self.triac_shorted and self.power_on:
            print(f"[AVR] Внимание: семистор пробит, некоторые команды могут быть ограничены")
        
        # AT+SS? - статус (режим)
        if cmd == "AT+SS?":
            if self.use_prefixes:
                return f"{self.mode_prefix}{self.mode}"
            else:
                return self.mode
        
        # AT+VO? - текущая мощность
        elif cmd == "AT+VO?":
            if self.use_prefixes:
                return f"{self.current_prefix}{self.current_power}"
            else:
                return str(self.current_power)
        
        # AT+VS? - целевая мощность
        elif cmd == "AT+VS?":
            if self.use_prefixes:
                return f"{self.target_prefix}{self.target_power}"
            else:
                return str(self.target_power)
        
        # AT+VS=xxx - установить мощность
        elif cmd.startswith("AT+VS="):
            match = re.match(r"AT\+VS=(\d{2,4})", cmd)  # До 4 цифр для мощности
            if match:
                power = int(match.group(1))
                
                # Если семистор пробит, ограничиваем максимальную мощность
                if self.triac_shorted:
                    max_power = self.triac_max_power
                else:
                    max_power = self.normal_max_power
                
                if 0 <= power <= max_power:
                    self.target_power = power
                    if self.power_on and power > 0:
                        # Если семистор пробит, устанавливаем максимальную мощность
                        if self.triac_shorted:
                            self.current_power = self.triac_max_power
                        else:
                            self.current_power = power
                        self.mode = "0"  # Работа
                    elif power == 0:
                        self.current_power = 0
                        self.mode = "2"  # Сон
                    return None  # Stab AVR не отвечает на команду установки
                else:
                    return "error"
            return "error"
        
        # AT+ON? - состояние питания
        elif cmd == "AT+ON?":
            return "ON" if self.power_on else "OFF"
        
        # AT+ON=x - включить/выключить
        elif cmd.startswith("AT+ON="):
            match = re.match(r"AT\+ON=(\d)", cmd)
            if match:
                state = int(match.group(1))
                self.power_on = (state == 1)
                if self.power_on:
                    self.mode = "1"  # Разгон при включении
                    # Если семистор пробит - максимальная мощность
                    if self.triac_shorted:
                        self.current_power = self.triac_max_power
                        self.target_power = self.triac_max_power
                    else:
                        self.current_power = self.normal_max_power
                        self.target_power = self.normal_max_power
                else:
                    self.mode = "2"  # Сон
                    self.current_power = 0
                    self.target_power = 0
                return None  # Stab AVR не отвечает на команду включения/выключения
            return "error"
        
        else:
            return None