"""
Эмулятор UDP стабилизатора
"""
import sys
import time
import socket
import threading
import urllib.parse
import random
from datetime import datetime
from PySide6.QtCore import QMetaObject, Qt, Slot, Q_ARG
from emulators.base import BaseEmulator
from utils import get_timestamp

class UDPStabEmulator(BaseEmulator):
    """Эмулятор UDP стабилизатора (с реальной UDP рассылкой)"""
    def __init__(self):
        super().__init__()
        self.device_type = "UDP_STAB"
        self.current_power = 0
        self.target_power = 0
        self.heat_enabled = False
        self.boost_mode = False
        self.error_mode = False
        self.triac_shorted = False
        
        # Для имитации ошибок
        self.error_start_time = 0
        
        # Для имитации стабилизации
        self.stabilization_start_time = None
        self.stabilization_phase = "idle"  # idle, rapid, slow, stable
        self.rapid_target = 0  # Цель для быстрого приближения
        self.stabilization_counter = 0
        
        # Инициализируем список интерфейсов
        self.available_interfaces = []
        self.current_interface_index = 0
        
        # Первоначальное обнаружение интерфейсов
        self._discover_interfaces()
        
        self.udp_port = 12345
        self.udp_socket = None
        self.udp_thread = None
        self.running = False
        self.last_broadcast_time = 0
        self.broadcast_interval = 1000  # мс
             
        # Для обработки HTTP запросов
        self.last_http_request = None
        self.http_request_count = 0
        
        # HTTP сервер эмуляции (для обработки команд)
        self.http_commands = {
            "/set_power": self.handle_set_power,
            "/set_heat": self.handle_set_heat,
            "/set_boost": self.handle_set_boost,
        }
        # TCP сервер для HTTP
        self.tcp_server = None
        self.tcp_thread = None
        self.tcp_running = False
        self.tcp_port = 80  # HTTP порт
        
        # Для хранения состояния перед разгоном
        self.pre_boost_state = {
            'target_power': 0,
            'heat_enabled': False
        }
        
        # Настройки для всех эмуляторов (глобальные)
        self.emulator_settings = {
            'simulate_triac_failure': False,  # Галка в GUI
        }

    def set_parent(self, parent):
        """Установить родительское окно"""
        self.parent = parent
        
        # При установке родителя, обновляем IP в UI если нужно
        if parent and hasattr(parent, 'emulator_configs'):
            # Обновляем IP в конфиге, чтобы UI показал правильный IP
            if "UDP_STAB" in parent.emulator_configs:
                parent.emulator_configs["UDP_STAB"]["ip"] = self.ip_address

    def _apply_triac_state(self):
        """Применить состояние семистора к текущей мощности"""
        if self.triac_shorted and self.heat_enabled:
            # Если семистор пробит и нагрев включен, мощность всегда 3000
            self.current_power = 3000
            self.target_power = 3000
            print(f"[UDP] Применено состояние пробитого семистора: мощность 3000W")
        elif not self.triac_shorted and self.heat_enabled and not self.boost_mode:
            # Если семистор исправен и нагрев включен (не в режиме разгона)
            self.current_power = self.target_power
            print(f"[UDP] Нормальный режим: мощность {self.current_power}W")

    def _create_http_response(self, status_code=200, body="OK", content_type="text/plain"):
        """Создать HTTP ответ с учетом проблем ESP32"""
        body_bytes = body.encode('utf-8')
        response = f"HTTP/1.1 {status_code} {'OK' if status_code == 200 else 'Bad Request'}\r\n"
        response += f"Content-Type: {content_type}\r\n"
        response += f"Content-Length: {len(body_bytes)}\r\n"
        response += "Connection: keep-alive\r\n"
        response += "Keep-Alive: timeout=5, max=1\r\n"
        response += "\r\n"
        response += body
        return response
    
    def _discover_interfaces(self):
        """Обнаружить все сетевые интерфейсы"""
        self.available_interfaces = []
        
        # Метод 1: Через psutil (лучший вариант)
        interfaces1 = self._get_interfaces_via_psutil()
        if interfaces1:
            self.available_interfaces.extend(interfaces1)
        
        # Метод 2: Через netifaces
        interfaces2 = self._get_interfaces_via_netifaces()
        if interfaces2:
            # Добавляем только уникальные интерфейсы
            for iface in interfaces2:
                if not any(i['ip'] == iface['ip'] for i in self.available_interfaces):
                    self.available_interfaces.append(iface)
        
        # Метод 3: Fallback через socket
        if not self.available_interfaces:
            interfaces3 = self._get_interfaces_via_socket()
            if interfaces3:
                self.available_interfaces.extend(interfaces3)
        
        # Сортируем интерфейсы по приоритету:
        # 1. Беспроводные (WiFi, hotspot)
        # 2. Ethernet
        # 3. Другие
        def interface_priority(iface):
            name = iface['name'].lower()
            if 'wireless' in name or 'wifi' in name or 'wlan' in name or 'wi-fi' in name:
                return 0
            elif 'local' in name or 'hotspot' in name or 'ap' in name:
                return 1
            elif 'ethernet' in name or 'eth' in name or 'lan' in name:
                return 2
            elif 'virtual' in name or 'vmware' in name or 'virtualbox' in name:
                return 9
            else:
                return 3
        
        self.available_interfaces.sort(key=interface_priority)
        
        # Устанавливаем первый интерфейс как текущий
        if self.available_interfaces:
            self.current_interface_index = 0
            self.ip_address = self.available_interfaces[0]['ip']
            # Используем правильный broadcast как в реальном коде: последний октет = 255
            ip_parts = self.ip_address.split('.')
            if len(ip_parts) == 4:
                self.broadcast_ip = f"{ip_parts[0]}.{ip_parts[1]}.{ip_parts[2]}.255"
            else:
                self.broadcast_ip = "255.255.255.255"
        else:
            # Fallback
            self.ip_address = "192.168.1.100"
            self.broadcast_ip = "255.255.255.255"
        
        print(f"[UDP] Обнаружено интерфейсов: {len(self.available_interfaces)}")
        for i, iface in enumerate(self.available_interfaces):
            print(f"  {i+1}. {iface['name']}: {iface['ip']} -> {iface['broadcast']}")
    
    def _get_interfaces_via_psutil(self):
        """Получить интерфейсы через psutil (самый надежный метод)"""
        interfaces = []
        
        try:
            import psutil
            
            # Получаем все сетевые интерфейсы
            for iface_name, addrs in psutil.net_if_addrs().items():
                ipv4_addrs = []
                
                # Ищем IPv4 адреса
                for addr in addrs:
                    if addr.family == socket.AF_INET:
                        ipv4_addrs.append({
                            'address': addr.address,
                            'netmask': addr.netmask,
                            'broadcast': addr.broadcast
                        })
                
                # Обрабатываем найденные IPv4 адреса
                for addr_info in ipv4_addrs:
                    ip = addr_info['address']
                    
                    # Пропускаем невалидные адреса
                    if (ip == '127.0.0.1' or 
                        ip.startswith('169.254.') or  # APIPA
                        ip.startswith('0.0.0.0')):
                        continue
                    
                    # Определяем broadcast как в реальном коде: последний октет = 255
                    ip_parts = ip.split('.')
                    if len(ip_parts) == 4:
                        broadcast = f"{ip_parts[0]}.{ip_parts[1]}.{ip_parts[2]}.255"
                    else:
                        broadcast = '255.255.255.255'
                    
                    # Определяем тип интерфейса
                    iface_type = "unknown"
                    if 'Wireless' in iface_name or 'WiFi' in iface_name or 'WLAN' in iface_name:
                        iface_type = "wifi"
                    elif 'Ethernet' in iface_name or 'ETH' in iface_name:
                        iface_type = "ethernet"
                    elif 'Local' in iface_name or 'hotspot' in iface_name:
                        iface_type = "hotspot"
                    elif 'Virtual' in iface_name or 'VMware' in iface_name:
                        iface_type = "virtual"
                    
                    interfaces.append({
                        'name': iface_name,
                        'ip': ip,
                        'netmask': addr_info.get('netmask', '255.255.255.0'),
                        'broadcast': broadcast,
                        'type': iface_type,
                        'description': f"{iface_name} ({iface_type}): {ip}"
                    })
            
        except ImportError:
            print("[UDP] psutil не установлен. Пропускаем этот метод.")
        except Exception as e:
            print(f"[UDP] Ошибка при использовании psutil: {e}")
        
        return interfaces
    
    def _get_interfaces_via_netifaces(self):
        """Получить интерфейсы через netifaces"""
        interfaces = []
        
        try:
            import netifaces
            
            for iface_name in netifaces.interfaces():
                try:
                    addrs = netifaces.ifaddresses(iface_name)
                    if netifaces.AF_INET in addrs:
                        for addr_info in addrs[netifaces.AF_INET]:
                            ip = addr_info.get('addr', '')
                            
                            # Пропускаем невалидные адреса
                            if (not ip or ip == '127.0.0.1' or 
                                ip.startswith('169.254.') or
                                ip.startswith('0.0.0.0')):
                                continue
                            
                            # Определяем broadcast как в реальном коде: последний октет = 255
                            ip_parts = ip.split('.')
                            if len(ip_parts) == 4:
                                broadcast = f"{ip_parts[0]}.{ip_parts[1]}.{ip_parts[2]}.255"
                            else:
                                broadcast = '255.255.255.255'
                            
                            # Определяем тип интерфейса
                            iface_type = "unknown"
                            if 'wl' in iface_name.lower() or 'wifi' in iface_name.lower():
                                iface_type = "wifi"
                            elif 'eth' in iface_name.lower() or 'lan' in iface_name.lower():
                                iface_type = "ethernet"
                            elif 'local' in iface_name.lower() or 'hotspot' in iface_name.lower():
                                iface_type = "hotspot"
                            
                            interfaces.append({
                                'name': iface_name,
                                'ip': ip,
                                'netmask': addr_info.get('netmask', '255.255.255.0'),
                                'broadcast': broadcast,
                                'type': iface_type,
                                'description': f"{iface_name} ({iface_type}): {ip}"
                            })
                except:
                    continue
        
        except ImportError:
            print("[UDP] netifaces не установлен. Пропускаем этот метод.")
        except Exception as e:
            print(f"[UDP] Ошибка при использовании netifaces: {e}")
        
        return interfaces
    
    def _get_interfaces_via_socket(self):
        """Fallback метод получения интерфейсов через socket"""
        interfaces = []
        
        try:
            # Пытаемся получить все возможные интерфейсы
            hostname = socket.gethostname()
            
            # Пробуем разные методы получения IP
            ip_candidates = []
            
            # Метод 1: Через gethostbyname_ex
            try:
                _, _, ip_list = socket.gethostbyname_ex(hostname)
                ip_candidates.extend(ip_list)
            except:
                pass
            
            # Метод 2: Через подключение к внешнему серверу
            try:
                s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
                s.connect(("8.8.8.8", 80))
                local_ip = s.getsockname()[0]
                s.close()
                if local_ip and local_ip != '127.0.0.1':
                    ip_candidates.append(local_ip)
            except:
                pass
            
            # Обрабатываем найденные IP
            for ip in set(ip_candidates):
                if ip and ip != '127.0.0.1' and not ip.startswith('169.254.'):
                    octets = ip.split('.')
                    if len(octets) >= 3:
                        # Как в реальном коде: последний октет = 255
                        broadcast = f"{octets[0]}.{octets[1]}.{octets[2]}.255"
                        interfaces.append({
                            'name': 'primary',
                            'ip': ip,
                            'netmask': '255.255.255.0',
                            'broadcast': broadcast,
                            'type': 'detected',
                            'description': f"Обнаруженный: {ip}"
                        })
        
        except Exception as e:
            print(f"[UDP] Ошибка fallback метода: {e}")
        
        return interfaces
    
    def switch_to_next_interface(self):
        """Переключиться на следующий сетевой интерфейс"""
        if len(self.available_interfaces) <= 1:
            return False
        
        old_index = self.current_interface_index
        old_ip = self.ip_address
        
        # Переключаемся на следующий интерфейс
        self.current_interface_index = (self.current_interface_index + 1) % len(self.available_interfaces)
        interface = self.available_interfaces[self.current_interface_index]
        
        self.ip_address = interface['ip']
        # Используем правильный broadcast как в реальном коде
        ip_parts = self.ip_address.split('.')
        if len(ip_parts) == 4:
            self.broadcast_ip = f"{ip_parts[0]}.{ip_parts[1]}.{ip_parts[2]}.255"
        else:
            self.broadcast_ip = '255.255.255.255'
        
        print(f"[UDP] Переключено с интерфейса {old_index+1} на {self.current_interface_index+1}")
        print(f"[UDP] IP: {old_ip} -> {self.ip_address}")
        print(f"[UDP] Broadcast: {self.broadcast_ip}")
        
        # Перезапускаем рассылку если была активна
        was_running = self.running
        if was_running:
            self.stop()
            time.sleep(0.2)  # Даем время на остановку
        
        if was_running and self.enabled:
            self.start()
        
        return True
    
    def start(self):
        """Запуск UDP рассылки"""
        if not self.enabled or self.running:
            return
        
        print(f"[UDP] Запуск рассылки на {self.ip_address} -> {self.broadcast_ip}:{self.udp_port}")
        
        try:
            # Создаем сокет для широковещательной рассылки
            self.udp_socket = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
            self.udp_socket.setsockopt(socket.SOL_SOCKET, socket.SO_BROADCAST, 1)
            self.udp_socket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
            
            # Для Windows нужна дополнительная настройка
            if sys.platform == 'win32':
                self.udp_socket.setsockopt(socket.SOL_SOCKET, socket.SO_EXCLUSIVEADDRUSE, 0)
            
            self.running = True
            self.udp_thread = threading.Thread(target=self._broadcast_thread, daemon=True)
            self.udp_thread.start()
            
            # ЗАПУСКАЕМ TCP СЕРВЕР
            self._start_tcp_server()
            
            if self.parent:
                timestamp = get_timestamp(self.parent)
                QMetaObject.invokeMethod(self.parent.terminal, "append", 
                                        Qt.QueuedConnection,
                                        Q_ARG(str, f"{timestamp}[UDP] Рассылка запущена на порту {self.udp_port}"))
            
        except Exception as e:
            print(f"[UDP] Ошибка запуска: {e}")
            if self.parent:
                timestamp = get_timestamp(self.parent)
                QMetaObject.invokeMethod(self.parent.terminal, "append", 
                                        Qt.QueuedConnection,
                                        Q_ARG(str, f"{timestamp}[UDP] Ошибка запуска: {e}"))
    
    def _start_tcp_server(self):
        """Запуск TCP сервера для HTTP"""
        try:
            self.tcp_server = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            self.tcp_server.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
            self.tcp_server.bind(('0.0.0.0', self.tcp_port))
            self.tcp_server.listen(5)  # До 5 соединений в очереди
            self.tcp_server.settimeout(1.0)  # Таймаут для accept
            
            self.tcp_running = True
            self.tcp_thread = threading.Thread(target=self._tcp_server_thread, daemon=True)
            self.tcp_thread.start()
            
            print(f"[TCP] Сервер запущен на порту {self.tcp_port}")
            
        except Exception as e:
            print(f"[TCP] Ошибка запуска сервера: {e}")
    
    def _tcp_server_thread(self):
        """Поток TCP сервера"""
        print("[TCP] Поток сервера запущен")
        
        while self.tcp_running and self.tcp_server:
            try:
                # Ждем подключения
                client_socket, client_address = self.tcp_server.accept()
                client_socket.settimeout(5.0)
                
                # Обрабатываем подключение в отдельном потоке
                client_thread = threading.Thread(
                    target=self._handle_tcp_client,
                    args=(client_socket, client_address),
                    daemon=True
                )
                client_thread.start()
                
            except socket.timeout:
                continue  # Таймаут accept - нормально, продолжаем
            except Exception as e:
                if self.tcp_running:  # Игнорируем ошибки при остановке
                    print(f"[TCP] Ошибка accept: {e}")
                time.sleep(0.1)
        
        print("[TCP] Поток сервера завершен")
    
    def _handle_tcp_client(self, client_socket, client_address):
        """Обработка TCP клиента"""
        import time
        start_time = time.time()
        client_ip, client_port = client_address
        print(f"[TCP] Новое подключение от {client_ip}:{client_port}")
        
        try:
            # 1. Читаем заголовки HTTP запроса
            request_data = b""
            headers_complete = False
            content_length = 0
            
            while not headers_complete:
                chunk = client_socket.recv(1024)
                if not chunk:
                    return
                request_data += chunk
                
                # Ищем конец заголовков
                if b"\r\n\r\n" in request_data:
                    headers_part, rest = request_data.split(b"\r\n\r\n", 1)
                    request_data = headers_part + b"\r\n\r\n" + rest
                    headers_complete = True
                    
                    # Парсим заголовки, чтобы узнать Content-Length
                    headers_text = headers_part.decode('utf-8', errors='ignore')
                    for line in headers_text.split('\r\n'):
                        if line.lower().startswith('content-length:'):
                            try:
                                content_length = int(line.split(':')[1].strip())
                            except:
                                content_length = 0
            
            # 2. Если есть тело запроса (POST), дочитываем его
            if content_length > 0:
                # Уже прочитали часть тела в 'rest'
                body_data = rest if 'rest' in locals() else b""
                body_received = len(body_data)
                
                # Дополучаем оставшиеся данные
                while body_received < content_length:
                    remaining = content_length - body_received
                    chunk = client_socket.recv(min(1024, remaining))
                    if not chunk:
                        break
                    body_data += chunk
                    body_received += len(chunk)
                
                # Полный запрос
                full_request = headers_part.decode('utf-8', errors='ignore') + "\r\n\r\n" + body_data.decode('utf-8', errors='ignore')
            else:
                full_request = request_data.decode('utf-8', errors='ignore')
            
            print(f"[TCP] Полный запрос от {client_ip}:\n{full_request[:500]}...")
            
            # 3. Парсим запрос
            lines = full_request.split('\r\n')
            if len(lines) > 0:
                first_line = lines[0]
                parts = first_line.split()
                if len(parts) >= 3:
                    method = parts[0]
                    path = parts[1]
                    
                    # 4. Извлекаем параметры из URL (GET) или тела (POST)
                    params = {}
                    
                    if method.upper() == 'GET':
                        # Параметры в URL для GET
                        if '?' in path:
                            path_part, query_string = path.split('?', 1)
                            if ' ' in query_string:
                                query_string = query_string.split(' ', 1)[0]
                            self._parse_query_string(query_string, params)
                        else:
                            path_part = path.split(' ', 1)[0] if ' ' in path else path
                    
                    elif method.upper() == 'POST':
                        # Параметры в теле для POST
                        path_part = path.split(' ', 1)[0] if ' ' in path else path
                        
                        # Ищем тело запроса (после пустой строки)
                        body_start = False
                        request_body = ""
                        for line in lines:
                            if line == "":  # Пустая строка - конец заголовков
                                body_start = True
                                continue
                            if body_start:
                                request_body = line
                                break
                        
                        # Парсим тело как application/x-www-form-urlencoded
                        if request_body:
                            self._parse_query_string(request_body, params)
                    
                    # 5. Обрабатываем команду
                    response_body = ""
                    
                    if path_part == '/set_heat':
                        if 'heat' in params:
                            heat_val = params['heat']
                            old_state = self.heat_enabled
                            self.heat_enabled = (heat_val == '1')
                            # Сбрасываем стабилизацию
                            self.stabilization_phase = "idle"
                            self.stabilization_start_time = None
                            # Применяем логику семистора
                            if self.triac_shorted and self.heat_enabled:
                                self.current_power = 3000
                                print(f"[TCP] Нагрев включен с пробитым семистором: мощность 3000W")
                            elif not self.heat_enabled:
                                # При выключении нагрева:
                                self.error_mode = False
                                self.error_start_time = 0
                                self.current_power = 0
                                if self.boost_mode:
                                    self.boost_mode = False
                                    print(f"[TCP] Режим разгона выключен вместе с нагревом")
                            elif old_state == False and self.heat_enabled == True and not self.triac_shorted:
                                # При включении нагрева с нуля (семмистор исправен):
                                if self.target_power > 0:
                                    self.current_power = self.target_power
                                print(f"[TCP] Нагрев включен, мощность: {self.current_power}W")
                            
                            response_body = "OK"
                            print(f"[TCP] Установлен нагрев: {'ВКЛ' if self.heat_enabled else 'ВЫКЛ'}")
                        else:
                            response_body = "Missing 'heat' parameter"
                    elif path_part == '/set_power':
                        if 'power' in params:
                            try:
                                power = float(params['power'])
                                if 0 <= power <= 10000:
                                    power = int(power)
                                    if self.boost_mode:
                                        self.boost_mode = False
                                        print(f"[TCP] Режим разгона выключен при установке мощности")
                                    
                                    self.target_power = power
                                    
                                    # Сбрасываем стабилизацию при изменении мощности
                                    self.stabilization_phase = "idle"
                                    self.stabilization_start_time = None
                                    
                                    # Обновляем текущую мощность в зависимости от состояния семистора
                                    if self.heat_enabled and not self.boost_mode:
                                        if self.triac_shorted:
                                            # Если семистор пробит - текущая мощность всегда 3000
                                            self.current_power = 3000
                                            print(f"[TCP] Установлена целевая мощность: {power}W, но текущая 3000W (пробит семистор)")
                                        else:
                                            # Если семистор исправен - начинаем стабилизацию
                                            self.current_power = power * 0.8  # Начинаем с 80% от целевой для имитации
                                            self.stabilization_phase = "rapid"
                                            self.stabilization_start_time = time.time()
                                            self.rapid_target = power * 1.05
                                            print(f"[TCP] Установлена мощность: {power}W, начата стабилизация")
                                    
                                    # При установке мощности 0 - выключаем нагрев
                                    if power == 0:
                                        self.heat_enabled = False
                                        self.boost_mode = False
                                        self.current_power = 0
                                        self.stabilization_phase = "idle"
                                        print(f"[TCP] Мощность 0W - нагрев и разгон выключены")
                                    response_body = "OK"
                                else:
                                    response_body = "Power out of range"
                            except:
                                response_body = "Invalid power value"
                        else:
                            response_body = "Missing 'power' parameter"
                    elif path_part == '/set_boost':
                        if 'boost' in params:
                            boost_val = params['boost']
                            old_boost_state = self.boost_mode
                            
                            if boost_val == '1':
                                # ВКЛЮЧЕНИЕ режима разгона
                                if not self.boost_mode:
                                    # Сохраняем текущее состояние ПЕРЕД включением разгона
                                    self.pre_boost_state['target_power'] = self.target_power
                                    self.pre_boost_state['heat_enabled'] = self.heat_enabled
                                    
                                    self.boost_mode = True
                                    self.heat_enabled = True
                                    # Сбрасываем стабилизацию
                                    self.stabilization_phase = "idle"
                                    self.stabilization_start_time = None
                                    # В режиме разгона обе мощности 3000 (независимо от семистора)
                                    self.current_power = 3000
                                    self.target_power = 3000
                                    
                                    response_body = "OK"
                                    print(f"[TCP] Режим разгона ВКЛЮЧЕН, мощность: 3000W")
                                else:
                                    response_body = "OK"
                                    print(f"[TCP] Режим разгона уже включен")
                            else:  # boost_val == '0'
                                # ВЫКЛЮЧЕНИЕ режима разгона
                                if self.boost_mode:
                                    self.boost_mode = False
                                    
                                    # Восстанавливаем из сохраненного состояния
                                    self.target_power = self.pre_boost_state['target_power']
                                    self.heat_enabled = self.pre_boost_state['heat_enabled']
                                    
                                    # Восстанавливаем мощность с учетом семистора
                                    if self.triac_shorted and self.heat_enabled:
                                        self.current_power = 3000
                                        print(f"[TCP] Разгон ВЫКЛ. Пробит семистор: текущая 3000W, целевая {self.target_power}W")
                                    elif self.heat_enabled and not self.triac_shorted:
                                        self.current_power = self.target_power
                                        print(f"[TCP] Разгон ВЫКЛ. Восстановлено: текущая={self.current_power}W, целевая={self.target_power}W")
                                    else:
                                        self.current_power = 0
                                        print(f"[TCP] Разгон ВЫКЛ. Нагрев выключен")
                                    
                                    response_body = "OK"
                                else:
                                    response_body = "OK"
                                    print(f"[TCP] Режим разгона уже выключен")
                        else:
                            response_body = "Missing 'boost' parameter"      
                    elif path_part == '/set_triac':
                        if 'triac' in params:
                            triac_val = params['triac']
                            old_state = self.triac_shorted
                            self.triac_shorted = (triac_val == '1')
                            
                            # Логируем изменение
                            if self.triac_shorted != old_state:
                                print(f"[TCP] Состояние семистора: {'ПРОБИТ' if self.triac_shorted else 'НОРМА'}")
                            # Сбрасываем стабилизацию
                            self.stabilization_phase = "idle"
                            self.stabilization_start_time = None
                            # Применяем состояние семистора к мощности
                            if self.triac_shorted and self.heat_enabled:
                                # Включаем пробитый семистор - только текущая мощность 3000
                                self.current_power = 3000
                                print(f"[TCP] Установлена текущая мощность 3000W (пробит семистор), целевая: {self.target_power}W")
                            elif not self.triac_shorted and self.heat_enabled and not self.boost_mode:
                                # Выключаем пробитый семистор - текущая равна целевой
                                self.current_power = self.target_power
                                print(f"[TCP] Восстановлена нормальная мощность: {self.current_power}W")
                            
                            response_body = "OK"
                        else:
                            response_body = "Missing 'triac' parameter"
                    else:
                        response_body = "Not Found"
                    
                    end_time = time.time()
                    processing_time = (end_time - start_time) * 1000  # в мс
                    print(f"[TCP] Обработка заняла {processing_time:.1f} мс")
                    
                    # 6. Отправляем ответ
                    response = self._create_http_response(
                        200 if response_body == "OK" else 400,
                        response_body
                    )
                    client_socket.sendall(response.encode('utf-8'))   
                     
                    # ВАЖНО: Не закрываем соединение сразу
                    time.sleep(0.05)  # 50 мс задержки
                    print(f"[TCP] Отправлен ответ клиенту {client_ip}: {response_body}")
        
        except Exception as e:
            print(f"[TCP] Ошибка обработки клиента {client_ip}: {e}")
            import traceback
            traceback.print_exc()
        
        finally:
            client_socket.close()

    def _parse_query_string(self, query_string, params_dict):
        """Парсит строку запроса вида key1=value1&key2=value2"""
        try:
            for param in query_string.split('&'):
                if '=' in param:
                    key, value = param.split('=', 1)
                    # Декодируем URL-encoded символы
                    import urllib.parse
                    key = urllib.parse.unquote(key)
                    value = urllib.parse.unquote(value)
                    params_dict[key] = value
                elif param:  # Параметр без значения
                    params_dict[param] = ""
        except Exception as e:
            print(f"[TCP] Ошибка парсинга query string: {e}")
            
    def _create_http_response(self, status_code, body):
        """Создать HTTP ответ"""
        status_text = {
            200: "OK",
            400: "Bad Request",
            404: "Not Found"
        }.get(status_code, "Unknown")
        
        body_bytes = body.encode('utf-8')
        response = f"HTTP/1.1 {status_code} {status_text}\r\n"
        response += "Content-Type: text/plain\r\n"
        response += f"Content-Length: {len(body_bytes)}\r\n"
        response += "Connection: close\r\n"
        response += "\r\n"
        response += body
        
        return response
    
    def stop(self):
        """Остановка UDP рассылки и TCP сервера"""
        print("[UDP] Остановка сервера...")
        self.running = False
        self.tcp_running = False
        
        # Останавливаем TCP сервер
        if self.tcp_server:
            try:
                self.tcp_server.close()
            except:
                pass
            self.tcp_server = None
        
        if self.tcp_thread and self.tcp_thread.is_alive():
            self.tcp_thread.join(timeout=1.0)
        
        # Останавливаем UDP (существующий код)
        if self.udp_thread and self.udp_thread.is_alive():
            self.udp_thread.join(timeout=1.0)
        
        if self.udp_socket:
            try:
                self.udp_socket.close()
            except:
                pass
            self.udp_socket = None
        
        if self.parent:
            timestamp = get_timestamp()
            QMetaObject.invokeMethod(
                self.parent.terminal, 
                "append", 
                Qt.QueuedConnection,
                Q_ARG(str, f"{timestamp}[UDP] Сервер остановлен")
            )
    
    def set_enabled(self, enabled: bool):
        """Включить/выключить эмулятор"""
        if self.enabled == enabled:
            return
        
        self.enabled = enabled
        
        if enabled:
            self.start()
        else:
            self.stop()
        
        print(f"[{self.device_type}] Эмулятор {'включен' if enabled else 'выключен'}")
        
        if self.parent:
            timestamp = get_timestamp(self.parent)
            QMetaObject.invokeMethod(self.parent.terminal, "append", 
                                    Qt.QueuedConnection,
                                    Q_ARG(str, f"{timestamp}[UDP] Эмулятор {'включен' if enabled else 'выключен'}"))
    
    def _broadcast_thread(self):
        """Поток широковещательной рассылки UDP пакетов"""
        print("[UDP] Поток рассылки запущен")
        
        while self.running and self.udp_socket:
            try:
                current_time = time.time() * 1000
                
                if current_time - self.last_broadcast_time >= self.broadcast_interval:
                    # Формируем сообщение в формате стабилизатора (как в реальном коде)
                    mode = 0
                    if self.heat_enabled:
                        mode |= 0x01  # Бит 0: Нагрев включен
                    if self.boost_mode:
                        mode |= 0x02  # Бит 1: Режим разгона
                    if self.error_mode:
                        mode |= 0x04  # Бит 2: Ошибка стабилизатора
                    
                    # Формат: StPr|текущая_мощность|целевая_мощность|режим (как в реальном коде)
                    message = f"StPr|{int(self.current_power)}|{int(self.target_power)}|{mode}"
                    
                    # ОТЛАДОЧНЫЙ ВЫВОД
                    print(f"[UDP BROADCAST] triac_shorted={self.triac_shorted}, message={message}")
                    
                    try:
                        # Отправляем широковещательный пакет (как в реальном коде)
                        self.udp_socket.sendto(
                            message.encode('utf-8'),
                            (self.broadcast_ip, self.udp_port)
                        )
                        
                        # Также отправляем на localhost для тестирования
                        self.udp_socket.sendto(
                            message.encode('utf-8'),
                            ("127.0.0.1", self.udp_port)
                        )
                        
                        if self.parent:
                            timestamp = get_timestamp(self.parent)
                            log_msg = f"{timestamp}UDP >> {message} (на {self.broadcast_ip}:{self.udp_port})"
                            QMetaObject.invokeMethod(self.parent.terminal, "append", 
                                                    Qt.QueuedConnection,
                                                    Q_ARG(str, log_msg))
                        
                        self.last_broadcast_time = current_time
                        
                    except Exception as e:
                        print(f"[UDP] Ошибка отправки: {e}")
                
                time.sleep(0.01)  # Небольшая пауза
                
            except Exception as e:
                print(f"[UDP] Ошибка в потоке рассылки: {e}")
                time.sleep(0.1)
        
        print("[UDP] Поток рассылки завершен")
    
    # Обработчики HTTP команд (как в реальном коде)
    def handle_set_power(self, params):
        """Обработка команды установки мощности"""
        if "power" in params:
            try:
                power = float(params["power"][0])
                if 0 <= power <= 10000:
                    self.target_power = power
                    # В реальном коде здесь была бы функция setPower()
                    self.heat_enabled = True
                    self.boost_mode = False
                    self.error_mode = False
                    return True
            except:
                pass
        return False
    
    def handle_set_heat(self, params):
        """Обработка команды включения/выключения нагрева"""
        if "heat" in params:
            try:
                heat_val = params["heat"][0]
                self.heat_enabled = (heat_val == "1")
                if not self.heat_enabled:
                    # В реальном коде здесь была бы функция heat()
                    pass
                return True
            except:
                pass
        return False
    
    def handle_set_boost(self, params):
        """Обработка команды включения/выключения режима разгона"""
        if "boost" in params:
            try:
                boost_val = params["boost"][0]
                self.boost_mode = (boost_val == "1")
                if self.boost_mode:
                    # В реальном коде здесь была бы функция boost()
                    self.target_power = 10000
                    self.heat_enabled = True
                return True
            except:
                pass
        return False
    
    def process_command(self, command: str):
        """Обработка команд UDP стабилизатора (HTTP команды)"""
        if not self.enabled:
            return None
        
        cmd = command.strip()
        print(f"[UDP] Обработка команды: {cmd}")
        
        # Эмуляция HTTP запросов (как в реальном коде)
        # Реальный код ожидает POST запросы вида: POST /set_power?power=1000 HTTP/1.1
        
        # Упрощенная обработка для COM-терминала
        if cmd.startswith("POST ") or cmd.startswith("GET "):
            # Это HTTP запрос
            parts = cmd.split()
            if len(parts) >= 2:
                method = parts[0]
                path = parts[1]
                
                # Логируем полученный запрос
                print(f"[UDP] HTTP {method} запрос: {path}")
                
                # Парсим параметры
                params = {}
                if "?" in path:
                    path_part = path.split("?")[0]
                    query_string = path.split("?")[1]
                    # Убираем возможные параметры после пробела (HTTP/1.1 и т.д.)
                    if " " in query_string:
                        query_string = query_string.split(" ")[0]
                    params = urllib.parse.parse_qs(query_string)
                else:
                    path_part = path.split(" ")[0] if " " in path else path
                
                # Обрабатываем команду
                handler = None
                for http_cmd, h in self.http_commands.items():
                    if path_part == http_cmd or path_part.startswith(http_cmd):
                        handler = h
                        break
                
                if handler:
                    if handler(params):
                        # В реальном коде ESP32 ожидает пустой ответ или редирект
                        # Возвращаем минимальный валидный HTTP ответ
                        response = "HTTP/1.1 200 OK\r\n"
                        response += "Content-Type: text/plain\r\n"
                        response += "Content-Length: 2\r\n"
                        response += "Connection: close\r\n"  # Меняем close на keep-alive
                        response += "\r\n"
                        response += "OK"
                        time.sleep(0.15)  # Небольшая задержка перед отправкой
                        return response
                    else:
                        # Ошибка в параметрах
                        response = "HTTP/1.1 400 Bad Request\r\n"
                        response += "Content-Type: text/plain\r\n"
                        response += "Content-Length: 5\r\n"
                        response += "Connection: close\r\n"
                        response += "\r\n"
                        response += "ERROR"
                        return response
                else:
                    # Неизвестный путь
                    response = "HTTP/1.1 404 Not Found\r\n"
                    response += "Content-Type: text/plain\r\n"
                    response += "Content-Length: 9\r\n"
                    response += "Connection: close\r\n"
                    response += "\r\n"
                    response += "Not Found"
                    return response
        
        # Также обрабатываем упрощенные команды для терминала
        elif cmd.startswith("power="):
            try:
                power = float(cmd[6:])
                if 0 <= power <= 10000:
                    # Если семистор пробит - игнорируем
                    if self.triac_shorted:
                        return "ERROR: Triac shorted, cannot change power"
                    
                    power = int(power)
                    self.target_power = power
                    
                    # Сбрасываем стабилизацию
                    self.stabilization_phase = "idle"
                    self.stabilization_start_time = None
                    
                    if not self.triac_shorted and self.heat_enabled:
                        # Начинаем стабилизацию
                        self.current_power = int(power * 0.8)
                        self.stabilization_phase = "rapid"
                        self.stabilization_start_time = time.time()
                        self.rapid_target = int(power * 1.05)
                    
                    self.heat_enabled = True
                    self.boost_mode = False
                    self.error_mode = False
                    return f"Power set to {power}W (stabilization started)"
            except:
                pass
            return "ERROR: Invalid power value"
        
        elif cmd.startswith("heat="):
            try:
                heat = int(cmd[5:])
                self.heat_enabled = (heat == 1)
                # Применяем состояние семистора
                if self.triac_shorted and self.heat_enabled:
                    self.current_power = 3000
                    self.target_power = 3000
                if not self.heat_enabled:
                    self.current_power = 0
                return f"Heat {'ON' if self.heat_enabled else 'OFF'}"
            except:
                pass
            return "ERROR: Invalid heat value"
        
        elif cmd.startswith("boost="):
            try:
                boost = int(cmd[6:])
                self.boost_mode = (boost == 1)
                if self.boost_mode:
                    self.current_power = 10000
                    self.heat_enabled = True
                return f"Boost {'ON' if self.boost_mode else 'OFF'}"
            except:
                pass
            return "ERROR: Invalid boost value"
        
        elif cmd == "status":
            # Возвращаем текущий статус
            mode = 0
            if self.heat_enabled:
                mode |= 0x01
            if self.boost_mode:
                mode |= 0x02
            if self.error_mode:
                mode |= 0x04
            
            return f"Power: {self.current_power:.1f}W / {self.target_power:.1f}W, " \
                f"Heat: {'ON' if self.heat_enabled else 'OFF'}, " \
                f"Boost: {'ON' if self.boost_mode else 'OFF'}, " \
                f"Error: {'YES' if self.error_mode else 'NO'}, " \
                f"Mode: 0x{mode:02X}"
        
        elif cmd == "reset":
            self.error_mode = False
            self.heat_enabled = False
            self.boost_mode = False
            self.current_power = 0
            self.target_power = 1000
            return "Reset complete"
        
        # Если команда не распознана, возвращаем пустой ответ чтобы не вызывать ресет
        return ""
    
    def force_power_update(self):
        """Принудительно обновить мощность в зависимости от состояния семистора"""
        if self.triac_shorted and self.heat_enabled:
            # Если семистор пробит и нагрев включен - ТОЛЬКО ТЕКУЩАЯ мощность 3000
            self.current_power = 3000
            # Целевая мощность НЕ меняется!
            print(f"[UDP FORCE] triac_shorted=True: current_power=3000W, target_power={self.target_power}W (не изменилась)")
        elif not self.triac_shorted and self.heat_enabled and not self.boost_mode:
            # Если семистор исправен - текущая мощность равна целевой
            self.current_power = self.target_power
            print(f"[UDP FORCE] triac_shorted=False: current_power={self.current_power}W, target_power={self.target_power}W")
    
    def periodic_update(self):
        """Периодическое обновление (имитация изменения состояния)"""
        if not self.enabled:
            return
        
        # Если семистор пробит и нагрев включен - ТОЛЬКО ТЕКУЩАЯ мощность 3000
        if self.triac_shorted and self.heat_enabled:
            if self.current_power != 3000:
                self.current_power = 3000
                # Сбрасываем стабилизацию при пробитом семисторе
                self.stabilization_phase = "idle"
                print(f"[UDP] Пробит семистор: текущая мощность 3000W, целевая {self.target_power}W")
            return
        elif self.triac_shorted and not self.heat_enabled:
            # Если семистор пробит, но нагрев выключен
            self.current_power = 0
            self.stabilization_phase = "idle"
        
        # Если нагрев выключен или авария
        if not self.heat_enabled or self.error_mode:
            self.current_power = 0
            self.stabilization_phase = "idle"
            return
        
        # Режим разгона - всегда 3000
        if self.boost_mode:
            if self.current_power != 3000:
                self.current_power = 3000
                self.target_power = 3000
                self.stabilization_phase = "idle"
                print(f"[UDP] Режим разгона: мощность 3000W")
            return
        
        # Имитация стабилизации мощности
        self._simulate_stabilization()
        
        # Имитация ошибок стабилизатора
        self._simulate_errors()

    def _simulate_stabilization(self):
        """Имитация процесса стабилизации мощности"""
        # Проверяем, нужна ли стабилизация
        if abs(self.current_power - self.target_power) > 10 and self.stabilization_phase == "idle":
            # Начинаем стабилизацию
            self.stabilization_phase = "rapid"
            self.stabilization_start_time = time.time()
            if self.current_power < self.target_power:
                self.rapid_target = int(self.target_power * 1.05)
            else:
                self.rapid_target = int(self.target_power * 0.95)
            print(f"[UDP] Начало стабилизации: цель {self.target_power}W, быстрый подход к {self.rapid_target}W")
        
        if self.stabilization_phase == "rapid":
            # Быстрое приближение к 95-105% от целевой
            diff = self.rapid_target - self.current_power
            
            # Рассчитываем изменение (минимум 1 Вт чтобы не застрять)
            if abs(diff) > 0:
                change = int(diff * 0.3)
                # Гарантируем минимальное изменение 1 Вт в нужном направлении
                if abs(change) < 1:
                    change = 1 if diff > 0 else -1
                
                self.current_power += change
                
                # Проверяем достижение цели быстрой стабилизации
                if abs(self.current_power - self.rapid_target) <= 10:
                    self.stabilization_phase = "slow"
                    print(f"[UDP] Медленная стабилизация: {self.current_power}W -> {self.target_power}W")
        
        elif self.stabilization_phase == "slow":
            # Медленная стабилизация в течение ~10 секунд
            elapsed = time.time() - self.stabilization_start_time
            if elapsed < 10:  # 10 секунд медленной стабилизации
                diff = self.target_power - self.current_power
                
                if abs(diff) > 0:
                    change = int(diff * 0.02)
                    # Гарантируем минимальное изменение 1 Вт
                    if abs(change) < 1:
                        change = 1 if diff > 0 else -1
                    
                    self.current_power += change
            else:
                self.stabilization_phase = "stable"
                print(f"[UDP] Стабилизировано: {self.current_power}W")
        
        elif self.stabilization_phase == "stable":
            # Стабильный режим с небольшими колебаниями ±2-3 Вт
            self.stabilization_counter += 1
            if self.stabilization_counter % 10 == 0:  # Каждые 10 обновлений
                # Небольшие случайные колебания
                fluctuation = random.randint(-3, 3)
                self.current_power = self.target_power + fluctuation
                # Держим в пределах ±5 Вт от целевой
                if abs(self.current_power - self.target_power) > 5:
                    self.current_power = self.target_power + (5 if fluctuation > 0 else -5)
        
        elif self.stabilization_phase == "idle" and self.heat_enabled:
            # Плавное изменение если разница небольшая (до 10 Вт)
            diff = self.target_power - self.current_power
            if abs(diff) > 1:
                change = int(diff * 0.1)
                # Гарантируем минимальное изменение
                if abs(change) < 1:
                    change = 1 if diff > 0 else -1
                self.current_power += change
    
    def _simulate_errors(self):
        """Имитация ошибок стабилизатора"""
        # Проверяем настройку имитации ошибок через parent
        if hasattr(self, 'parent') and self.parent:
            emulator_name = self.device_type
            if emulator_name in self.parent.emulator_configs:
                config = self.parent.emulator_configs[emulator_name]
                simulate_errors = config.get("simulate_errors", False)
                
                # Если в настройках включена имитация ошибок, но у нас не установлен флаг
                if simulate_errors and not self.error_mode:
                    # Включаем режим ошибки
                    self.error_mode = True
                    self.error_start_time = time.time()
                    self.heat_enabled = False
                    self.current_power = 0
                    self.target_power = 0
                    if hasattr(self, 'stabilization_phase'):
                        self.stabilization_phase = "idle"
                    print(f"[UDP] Режим ошибки ВКЛЮЧЕН (из настроек)")
                
                # Если в настройках выключена имитация ошибок, но у нас установлен флаг
                elif not simulate_errors and self.error_mode:
                    # Выключаем режим ошибки
                    self.error_mode = False
                    self.error_start_time = 0
                    print(f"[UDP] Режим ошибки ВЫКЛЮЧЕН (из настроек)")