"""
Эмулятор универсального протокола стабилизатора
"""
import time
from emulators.base import BaseEmulator
from utils import get_timestamp

class UniversalProtocolEmulator(BaseEmulator):
    """Эмулятор универсального протокола стабилизатора"""
    def __init__(self):
        super().__init__()
        self.device_type = "UNI_PROTOCOL"
        self.main_value = 0        # Основное значение (напряжение/ток/мощность)
        self.aux_value = 0         # Вспомогательное значение
        self.mode = 0              # Режим: 0-нормальный, 1-разгон, 2-выключен, 3-авария
        self.data_composition = 0b000001  # По умолчанию: основное=напряжение, вспомогательное=напряжение сети
        self.network_voltage = 220
        self.connected = True
        self.last_report_time = 0
        self.auto_report_interval = 1000  # мс
        
    def periodic_update(self):
        """Автоматическая отправка отчетов"""
        if not self.enabled:
            return
        
        current_time = time.time() * 1000
        if current_time - self.last_report_time >= self.auto_report_interval:
            report = self.generate_report()
            if self.parent and hasattr(self.parent, 'serial_manager'):
                try:
                    # Используем serial_manager для отправки
                    if self.parent.serial_manager.is_connected():
                        self.parent.serial_manager.send_text(report + "\r")
                        if self.parent:
                            timestamp = get_timestamp(self.parent)
                            self.parent.terminal.append(f"{timestamp}[Авто-отчет] UNI >> {report.strip()}")
                except Exception as e:
                    print(f"[UNI] Ошибка отправки авто-отчета: {e}")
            self.last_report_time = current_time
    
    def generate_report(self):
        """Генерация отчета в формате T + 12 hex символов"""
        # Формируем пакет: T + 12 hex символов
        # [0]: T
        # [1-2]: состав данных (data_composition)
        # [3-4]: режим (mode)
        # [5-8]: основное значение (main_value)
        # [9-12]: вспомогательное значение (aux_value)
        
        data_comp_hex = f"{self.data_composition:02X}"
        mode_hex = f"{self.mode:02X}"
        main_hex = f"{self.main_value:04X}"
        aux_hex = f"{self.aux_value:04X}"
        
        report = f"T{data_comp_hex}{mode_hex}{main_hex}{aux_hex}"
        return report
    
    def process_command(self, command: str):
        """Обработка команд универсального протокола"""
        if not self.enabled:
            return None
        
        cmd = command.strip()
        print(f"[UNI] Обработка команды: {cmd}")
        
        # Автоматическая отправка отчета при пустой команде
        if cmd == "":
            return self.generate_report()
        
        # Mx - установка режима
        elif cmd.startswith("M"):
            try:
                mode = int(cmd[1:])
                if 0 <= mode <= 3:
                    self.mode = mode
                    if mode == 2:  # Выключен
                        self.main_value = 0
                        self.aux_value = 0
                    elif mode == 1:  # Разгон
                        self.main_value = 10000  # Максимальная мощность
                        self.aux_value = 10000
                    return self.generate_report()
            except:
                pass
        
        # Pxxxx - установка мощности
        elif cmd.startswith("P"):
            try:
                power = int(cmd[1:], 16)  # Hex значение
                if 0 <= power <= 10000:
                    self.main_value = power
                    self.aux_value = power
                    self.data_composition = 0b000011  # Основное=мощность
                    self.mode = 0  # Рабочий режим
                    return self.generate_report()
            except:
                pass
        
        # Nxx - запрос вспомогательного параметра
        elif cmd.startswith("N"):
            try:
                param = int(cmd[1:], 16)
                # Устанавливаем состав данных в зависимости от запрашиваемого параметра
                if param == 0x01:  # Напряжение
                    self.data_composition = 0b000101
                    self.aux_value = self.network_voltage
                elif param == 0x02:  # Ток
                    self.data_composition = 0b000110
                    self.aux_value = int(self.main_value / self.network_voltage * 10) if self.network_voltage > 0 else 0
                elif param == 0x03:  # Мощность
                    self.data_composition = 0b000111
                    self.aux_value = self.main_value
                return self.generate_report()
            except:
                pass
        
        # Любая другая команда - возвращаем отчет
        return self.generate_report()