"""
Вкладка HC-12
"""
import re
import time
from PySide6.QtWidgets import *
from PySide6.QtCore import *
from PySide6.QtGui import *

def create_hc12_tab(parent):
    """Создать компактную вкладку HC-12"""
    tab = QWidget()
    tab.parent = parent
    #tab.setMaximumHeight(140)  # Фиксируем максимальную высоту
    
    # Главный layout
    main_layout = QVBoxLayout(tab)
    main_layout.setContentsMargins(3, 3, 3, 3)
    main_layout.setSpacing(2)  # Уменьшаем spacing
    
    # Инструкция
    info_label = QLabel("Для настройки HC-12: замкните SET на GND, подключитесь на 9600")
    info_label.setStyleSheet("color: #d00; font-weight: bold; font-size: 11px;")
    info_label.setMaximumHeight(16)
    info_label.setAlignment(Qt.AlignCenter)
    main_layout.addWidget(info_label)
    
    # Основное содержимое
    content_widget = QWidget()
    content_widget.setMaximumHeight(120)
    content_layout = QHBoxLayout(content_widget)
    content_layout.setContentsMargins(0, 0, 0, 0)
    content_layout.setSpacing(6)  # Уменьшаем spacing
    
    # ========== КОЛОНКА 1: РЕЖИМЫ И ПРЕДНАСТРОЙКИ ==========
    col1 = QVBoxLayout()
    col1.setSpacing(4)  # Уменьшаем spacing
    
    # Режим работы
    mode_layout = QHBoxLayout()
    mode_layout.setSpacing(3)
    
    mode_label = QLabel("Режим:")
    mode_label.setAlignment(Qt.AlignRight | Qt.AlignVCenter)
    mode_label.setMinimumWidth(45)
    mode_label.setMaximumHeight(20)
    mode_layout.addWidget(mode_label)
    
    parent.hc12_mode = QComboBox()
    parent.hc12_mode.setMaximumWidth(60)
    parent.hc12_mode.setMaximumHeight(22)
    
    mode_options = [
        ("FU1", "AT+FU1", "Стандартный режим"),
        ("FU2", "AT+FU2", "Прозрачная передача"),
        ("FU3", "AT+FU3", "Режим сна"),
        ("FU4", "AT+FU4", "Сон с таймаутом")
    ]
    
    for text, cmd, tip in mode_options:
        parent.hc12_mode.addItem(text, (cmd, tip))
    
    parent.hc12_mode.setCurrentIndex(0)
    parent.hc12_mode.setToolTip("Режим работы HC-12")
    mode_layout.addWidget(parent.hc12_mode)
    
    set_mode_btn = QPushButton("✅")
    set_mode_btn.setFixedSize(22, 22)
    set_mode_btn.setToolTip("Установить режим")
    set_mode_btn.clicked.connect(
        lambda: parent.send_hc12_command(parent.hc12_mode.currentData()[0]))
    mode_layout.addWidget(set_mode_btn)
    mode_layout.addStretch()
    col1.addLayout(mode_layout)
    
    # Преднастройки
    debug_btn = QPushButton("Режим отладки")
    debug_btn.setMaximumHeight(24)
    debug_btn.setMaximumWidth(140)
    debug_btn.setToolTip("Загрузить отладочные настройки\nСкорость: 115200, Канал: 080")
    debug_btn.clicked.connect(lambda: set_hc12_preset(parent, 115200, "080"))
    col1.addWidget(debug_btn)
    
    stab_btn = QPushButton("Управл. стабилизатором")
    stab_btn.setMaximumHeight(24)
    stab_btn.setMaximumWidth(140)
    stab_btn.setToolTip("Загрузить настройки для управления стабилизатором\nСкорость: 9600, Канал: 001")
    stab_btn.clicked.connect(lambda: set_hc12_preset(parent, 9600, "001"))
    col1.addWidget(stab_btn)
    
    col1.addStretch()
    content_layout.addLayout(col1, 1)
    
    # ========== КОЛОНКА 2: ОСНОВНЫЕ НАСТРОЙКИ ==========
    col2 = QVBoxLayout()
    col2.setSpacing(4)
    
    # Канал
    channel_layout = QHBoxLayout()
    channel_layout.setSpacing(3)
    
    channel_label = QLabel("Канал:")
    channel_label.setAlignment(Qt.AlignRight)# | Qt.AlignVCenter)
    channel_label.setMinimumWidth(70)
    channel_label.setMaximumHeight(20)
    channel_layout.addWidget(channel_label)
    
    parent.hc12_channel = QComboBox()
    parent.hc12_channel.addItems([f"{i:03d}" for i in range(1, 128)])
    parent.hc12_channel.setCurrentText("001")
    parent.hc12_channel.setMaximumWidth(70)
    parent.hc12_channel.setMaximumHeight(22)
    parent.hc12_channel.setToolTip("Канал передачи (001-127)")
    channel_layout.addWidget(parent.hc12_channel)
    
    set_channel_btn = QPushButton("📡")
    set_channel_btn.setFixedSize(22, 22)
    set_channel_btn.setToolTip("Установить канал")
    set_channel_btn.clicked.connect(
        lambda: parent.send_hc12_command(f"AT+C{parent.hc12_channel.currentText()}"))
    channel_layout.addWidget(set_channel_btn)
    channel_layout.addStretch()
    col2.addLayout(channel_layout)
    
    # Скорость
    baud_layout = QHBoxLayout()
    baud_layout.setSpacing(3)
    
    baud_label = QLabel("Скорость:")
    baud_label.setAlignment(Qt.AlignRight)# | Qt.AlignVCenter)
    baud_label.setMinimumWidth(70)
    baud_label.setMaximumHeight(20)
    baud_layout.addWidget(baud_label)
    
    parent.hc12_baud = QComboBox()
    parent.hc12_baud.addItems(["1200", "2400", "4800", "9600", "19200", "38400", "57600", "115200"])
    parent.hc12_baud.setCurrentText("9600")
    parent.hc12_baud.setMaximumWidth(70)
    parent.hc12_baud.setMaximumHeight(22)
    parent.hc12_baud.setToolTip("Скорость передачи (бод)")
    baud_layout.addWidget(parent.hc12_baud)
    
    set_baud_btn = QPushButton("⚡")
    set_baud_btn.setFixedSize(22, 22)
    set_baud_btn.setToolTip("Установить скорость")
    set_baud_btn.clicked.connect(
        lambda: parent.send_hc12_command(f"AT+B{parent.hc12_baud.currentText()}"))
    baud_layout.addWidget(set_baud_btn)
    baud_layout.addStretch()
    col2.addLayout(baud_layout)
    
    # Мощность
    power_layout = QHBoxLayout()
    power_layout.setSpacing(3)
    
    power_label = QLabel("Мощность:")
    power_label.setAlignment(Qt.AlignRight)# | Qt.AlignVCenter)
    power_label.setMinimumWidth(70)
    power_label.setMaximumHeight(20)
    power_layout.addWidget(power_label)
    
    parent.hc12_power = QComboBox()
    parent.hc12_power.setMinimumWidth(85)
    parent.hc12_power.setMaximumWidth(95)
    parent.hc12_power.setMaximumHeight(22)
    
    power_options = [
        ("P1 (-1 dBm)", "AT+P1", "0.8 мВт"),
        ("P2 (+2 dBm)", "AT+P2", "1.6 мВт"),
        ("P3 (+5 dBm)", "AT+P3", "3.2 мВт"),
        ("P4 (+8 dBm)", "AT+P4", "6.3 мВт"),
        ("P5 (+11 dBm)", "AT+P5", "12 мВт"),
        ("P6 (+14 dBm)", "AT+P6", "25 мВт"),
        ("P7 (+17 dBm)", "AT+P7", "50 мВт"),
        ("P8 (+20 dBm)", "AT+P8", "100 мВт")
    ]
    
    for text, cmd, desc in power_options:
        parent.hc12_power.addItem(text, (cmd, desc))
    
    parent.hc12_power.setCurrentIndex(7)
    parent.hc12_power.setToolTip("Мощность передатчика")
    power_layout.addWidget(parent.hc12_power)
    
    set_power_btn = QPushButton("📶")
    set_power_btn.setFixedSize(22, 22)
    set_power_btn.setToolTip("Установить мощность")
    set_power_btn.clicked.connect(
        lambda: parent.send_hc12_command(parent.hc12_power.currentData()[0]))
    power_layout.addWidget(set_power_btn)
    power_layout.addStretch()
    col2.addLayout(power_layout)
    
    col2.addStretch()
    content_layout.addLayout(col2, 1)
    
    # ========== КОЛОНКА 3: БЫСТРЫЕ КОМАНДЫ С ТЕКСТОМ ==========
    col3 = QVBoxLayout()
    col3.setSpacing(3)
    
    # Быстрые команды с текстом
    quick_commands = [
        ("AT+V", "📖", "Версия"),
        ("AT+RX", "🔍", "Конфиг"),
        ("AT+DEFAULT", "🔄", "Сброс"),
        ("AT+RESET", "♻", "Рестарт"),
    ]
    
    for cmd, icon, text in quick_commands:
        btn_layout = QHBoxLayout()
        btn_layout.setSpacing(3)
        
        btn = QPushButton(icon)
        btn.setFixedSize(22, 22)
        btn.setToolTip(f"{text}: {cmd}")
        btn.clicked.connect(lambda checked, c=cmd: parent.send_hc12_command(c))
        btn_layout.addWidget(btn)
        
        label = QLabel(text)
        label.setStyleSheet("font-size: 12px;")
        label.setMinimumWidth(35)
        label.setMaximumHeight(20)
        btn_layout.addWidget(label)
        
        btn_layout.addStretch()
        col3.addLayout(btn_layout)
    
    col3.addStretch()
    content_layout.addLayout(col3, 1)
    
    # ========== КОЛОНКА 4: ПУСТОЙ РАЗДЕЛИТЕЛЬ ==========
    spacer = QWidget()
    spacer.setMinimumWidth(5)
    content_layout.addWidget(spacer, 1)
    
    # Добавляем содержимое
    main_layout.addWidget(content_widget, 1)
    
    return tab

def set_hc12_preset(parent, baud_rate, channel):
    """Установить предустановленную конфигурацию HC-12"""
    def send_commands():
        # Устанавливаем скорость
        baud_text = str(baud_rate)
        index = parent.hc12_baud.findText(baud_text)
        if index >= 0:
            parent.hc12_baud.setCurrentIndex(index)
            parent.send_hc12_command(f"AT+B{baud_text}")
        
        # Пауза между командами
        time.sleep(1)
        
        # Устанавливаем канал
        index = parent.hc12_channel.findText(channel)
        if index >= 0:
            parent.hc12_channel.setCurrentIndex(index)
        else:
            parent.hc12_channel.setCurrentText(channel)
        parent.send_hc12_command(f"AT+C{channel}")
        
        # Пауза между командами
        time.sleep(1)
        
        timestamp = parent.get_timestamp()
        parent.terminal.append(f"{timestamp}HC-12 >> Установлен пресет: {baud_rate} бод, канал {channel}")
    
    # Запускаем в отдельном потоке, чтобы не блокировать UI
    import threading
    thread = threading.Thread(target=send_commands, daemon=True)
    thread.start()

def parse_hc12_response(response, parent):
    """Парсинг ответа HC-12"""
    try:
        if "OK+B" in response:  # Скорость
            for baud in ["1200", "2400", "4800", "9600", "19200", "38400", "57600", "115200"]:
                if baud in response:
                    index = parent.hc12_baud.findText(baud)
                    if index >= 0:
                        parent.hc12_baud.setCurrentIndex(index)
                        break
        
        if "OK+C" in response:  # Канал
            match = re.search(r'C(\d{3})', response)
            if match:
                channel = match.group(1)
                index = parent.hc12_channel.findText(channel)
                if index >= 0:
                    parent.hc12_channel.setCurrentIndex(index)
                else:
                    parent.hc12_channel.setCurrentText(channel)
        
        if "OK+P" in response:  # Мощность передатчика
            match = re.search(r'OK\+P(\d)', response)
            if match:
                power = match.group(1)
                for i in range(parent.hc12_power.count()):
                    item_text = parent.hc12_power.itemText(i)
                    if f"P{power}" in item_text:
                        parent.hc12_power.setCurrentIndex(i)
                        break
        
        if "OK+FU" in response:  # Режим работы
            match = re.search(r'OK\+FU(\d)', response)
            if match:
                mode_num = match.group(1)
                for i in range(parent.hc12_mode.count()):
                    if f"FU{mode_num}" in parent.hc12_mode.itemText(i):
                        parent.hc12_mode.setCurrentIndex(i)
                        break
        
    except Exception as e:
        print(f"[HC-12 PARSE ERROR] {e}")