"""
Вкладка Пресеты команд
"""
import json
import os
from PySide6.QtWidgets import *
from PySide6.QtCore import *
from PySide6.QtGui import *
from utils import get_timestamp  # Импортируем из utils

class PresetsTab(QWidget):
    def __init__(self, parent):
        super().__init__()
        self.parent = parent
        self.presets_file = "presets.json"
        self.presets = self.load_presets()
        
        self.setup_ui()
    
    def setup_ui(self):
        """Настройка интерфейса вкладки Пресеты"""
        layout = QVBoxLayout(self)
        layout.setContentsMargins(5, 5, 5, 5)
        layout.setSpacing(5)
        
        # Заголовок
        #title_label = QLabel("Пресеты команд")
        #title_label.setStyleSheet("font-weight: bold; font-size: 12px; color: #4a9eff;")
        #title_label.setAlignment(Qt.AlignCenter)
        #layout.addWidget(title_label)
        
        # Основное содержимое в 3 колонки (по 3 пресета в каждой)
        content_widget = QWidget()
        content_layout = QHBoxLayout(content_widget)
        content_layout.setContentsMargins(0, 0, 0, 0)
        content_layout.setSpacing(10)
        
        # Создаем 3 колонки (всего 9 пресетов)
        for col_idx in range(3):
            col = QVBoxLayout()
            col.setSpacing(2)
            
            # 3 пресета в каждой колонке
            for row_idx in range(3):
                preset_idx = col_idx * 3 + row_idx
                preset_widget = self.create_preset_widget(preset_idx)
                col.addWidget(preset_widget)
            
            col.addStretch()
            content_layout.addLayout(col)
        
        layout.addWidget(content_widget, 1)
        
        # Кнопки управления - в одну строку в конце
        button_layout = QHBoxLayout()
        button_layout.setSpacing(10)
        
        # Кнопка Сохранить
        save_btn = QPushButton("💾 Сохранить")
        save_btn.setMinimumHeight(22)
        save_btn.clicked.connect(self.save_all_presets)
        button_layout.addWidget(save_btn)
        
        # Кнопка Очистить
        clear_btn = QPushButton("🗑 Очистить")
        clear_btn.setMinimumHeight(22)
        clear_btn.clicked.connect(self.clear_all_presets)
        button_layout.addWidget(clear_btn)
        
        button_layout.addStretch()
        layout.addLayout(button_layout)
    
    def create_preset_widget(self, preset_idx):
        """Создать виджет для одного пресета"""
        widget = QFrame()
        widget.setFrameShape(QFrame.Box)
        widget.setLineWidth(1)
        widget.setStyleSheet("""
            QFrame {
                border-radius: 1px;
            }
        """)
        
        layout = QHBoxLayout(widget)
        layout.setContentsMargins(2, 2, 2, 2)
        layout.setSpacing(2)
        
        # Поле для команды
        cmd_input = QLineEdit()
        cmd_input.setPlaceholderText(f"Команда {preset_idx + 1}")
        cmd_input.setProperty("preset_idx", preset_idx)
        cmd_input.setProperty("field", "command")
        cmd_input.setText(self.presets.get(str(preset_idx), {}).get("command", ""))
        cmd_input.textChanged.connect(self.update_preset_field)
        cmd_input.setMinimumHeight(22)
        layout.addWidget(cmd_input, 1)  # Растягиваем поле
        
        # Кнопка отправки (иконка)
        send_btn = QPushButton("📤")
        send_btn.setProperty("preset_idx", preset_idx)
        send_btn.clicked.connect(self.send_preset_command)
        send_btn.setFixedSize(30, 22)
        send_btn.setToolTip(f"Отправить команду пресета {preset_idx + 1}")
        layout.addWidget(send_btn)
        
        return widget
    
    def update_preset_field(self, text):
        """Обновить поле пресета"""
        sender = self.sender()
        preset_idx = sender.property("preset_idx")
        field = sender.property("field")
        
        if preset_idx is not None and field:
            preset_key = str(preset_idx)
            if preset_key not in self.presets:
                self.presets[preset_key] = {}
            
            self.presets[preset_key][field] = text
    
    def send_preset_command(self):
        """Отправить команду из пресета"""
        btn = self.sender()
        preset_idx = btn.property("preset_idx")
        
        if preset_idx is not None:
            preset_key = str(preset_idx)
            preset = self.presets.get(preset_key, {})
            command = preset.get("command", "")
            
            if command:
                # Отправляем команду через родительский терминал
                if self.parent and hasattr(self.parent, 'send_manual_command'):
                    self.parent.send_manual_command(command)
                
                # Используем функцию get_timestamp из utils
                timestamp = get_timestamp(self.parent)
                log_msg = f"{timestamp}[Пресет {preset_idx + 1}] {command}"
                
                if hasattr(self.parent, 'terminal'):
                    self.parent.terminal.append(log_msg)
    
    def load_presets(self):
        """Загрузить пресеты из файла"""
        if os.path.exists(self.presets_file):
            try:
                with open(self.presets_file, 'r', encoding='utf-8') as f:
                    return json.load(f)
            except:
                return {}
        return {}
    
    def save_presets(self):
        """Сохранить пресеты в файл"""
        try:
            with open(self.presets_file, 'w', encoding='utf-8') as f:
                json.dump(self.presets, f, ensure_ascii=False, indent=2)
            return True
        except Exception as e:
            print(f"[PRESETS] Ошибка сохранения: {e}")
            return False
    
    def save_all_presets(self):
        """Сохранить все пресеты"""
        if self.save_presets():
            # Используем функцию get_timestamp из utils
            timestamp = get_timestamp(self.parent)
            self.parent.terminal.append(f"{timestamp}[Пресеты] Сохранены в {self.presets_file}")
        
    def clear_all_presets(self):
        """Очистить все пресеты"""
        self.presets = {}
        
        # Очищаем UI
        for widget in self.findChildren(QLineEdit):
            if widget.property("preset_idx") is not None:
                widget.clear()
        
        # Используем функцию get_timestamp из utils
        timestamp = get_timestamp(self.parent)
        self.parent.terminal.append(f"{timestamp}[Пресеты] Все пресеты очищены")