"""
Менеджер COM порта
"""
import serial
import threading
import time
from datetime import datetime
from PySide6.QtCore import QObject, Signal, Slot

class SerialManager(QObject):
    data_received = Signal(str)
    binary_received = Signal(bytes)
    status_changed = Signal(str)
    
    def __init__(self, parent=None):
        super().__init__(parent)
        self.parent = parent
        self.serial_port = None
        self.read_thread = None
        self.running = False
        self.rx_count = 0
        self.tx_count = 0
        
    def scan_ports(self):
        """Сканирование доступных портов"""
        import serial.tools.list_ports
        
        ports = list(serial.tools.list_ports.comports())
        return [(port.device, port.description) for port in ports]
    
    def connect(self, port_name, baud_rate):
        """Подключение к COM порту"""
        try:
            self.serial_port = serial.Serial(
                port=port_name,
                baudrate=baud_rate,
                bytesize=serial.EIGHTBITS,
                parity=serial.PARITY_NONE,
                stopbits=serial.STOPBITS_ONE,
                timeout=0.1,
                write_timeout=1.0,
                xonxoff=False,
                rtscts=False,
                dsrdtr=False
            )
            
            # Сбрасываем буферы
            self.serial_port.reset_input_buffer()
            self.serial_port.reset_output_buffer()
            
            # Запускаем поток чтения
            self.running = True
            self.read_thread = threading.Thread(target=self.read_thread_func, daemon=True)
            self.read_thread.start()
            
            self.status_changed.emit(f"✓ Подключено к {port_name}")
            return True
            
        except Exception as e:
            self.status_changed.emit(f"ОШИБКА: {str(e)}")
            return False
    
    def disconnect(self):
        """Отключение от COM порта"""
        self.running = False
        
        if self.serial_port and self.serial_port.is_open:
            try:
                self.serial_port.close()
            except:
                pass
        
        self.serial_port = None
        
        if self.read_thread and self.read_thread.is_alive():
            self.read_thread.join(timeout=1.0)
        
        self.status_changed.emit("Отключено")
    
    def is_connected(self):
        """Проверка подключения"""
        return self.serial_port and self.serial_port.is_open
    
    def read_thread_func(self):
        """Поток чтения данных"""
        buffer = bytearray()
        
        while self.running and self.serial_port and self.serial_port.is_open:
            try:
                if self.serial_port.in_waiting > 0:
                    data = self.serial_port.read(self.serial_port.in_waiting)
                    
                    if data:
                        buffer.extend(data)
                        
                        # Обрабатываем буфер
                        while True:
                            # Проверяем какой эмулятор активен
                            if (self.parent and 
                                hasattr(self.parent, 'emulator_manager') and
                                self.parent.emulator_manager.current_emulator == "MODBUS_RTU"):
                                # Modbus RTU - обрабатываем бинарные данные
                                if len(buffer) >= 8:
                                    frame_data = bytes(buffer[:8])
                                    self.binary_received.emit(frame_data)
                                    buffer = buffer[8:]
                                else:
                                    break
                            else:
                                # Текстовые протоколы
                                if b'\r' in buffer:
                                    idx = buffer.index(b'\r')
                                elif b'\n' in buffer:
                                    idx = buffer.index(b'\n')
                                else:
                                    break
                                
                                line_bytes = buffer[:idx]
                                buffer = buffer[idx+1:]
                                
                                try:
                                    line = line_bytes.decode('utf-8', errors='ignore')
                                except:
                                    try:
                                        line = line_bytes.decode('cp1251', errors='ignore')
                                    except:
                                        line = line_bytes.decode('latin-1', errors='ignore')
                                
                                line = line.strip()
                                if line:
                                    self.data_received.emit(line)
                
                time.sleep(0.01)
                
            except Exception as e:
                print(f"[READ ERROR] {e}")
                time.sleep(0.1)
    
    def send_text(self, text, add_cr=False, add_lf=False):
        """Отправка текста"""
        if not self.is_connected():
            return False
        
        try:
            data = text
            if add_cr:
                data += "\r"
            if add_lf:
                data += "\n"
            
            self.serial_port.write(data.encode('utf-8'))
            self.serial_port.flush()
            self.tx_count += 1
            return True
            
        except Exception as e:
            print(f"[SEND ERROR] {e}")
            return False
    
    def send_binary(self, data):
        """Отправка бинарных данных"""
        if not self.is_connected():
            return False
        
        try:
            self.serial_port.write(data)
            self.serial_port.flush()
            self.tx_count += 1
            return True
            
        except Exception as e:
            print(f"[SEND ERROR] {e}")
            return False