"""
Базовый класс вкладки эмулятора
"""
from PySide6.QtWidgets import *
from PySide6.QtCore import *
from PySide6.QtGui import *
from utils import get_timestamp

class EmulatorTabBase(QWidget):
    """Базовый класс для всех вкладок эмуляторов"""
    
    def __init__(self, parent, emulator_name):
        super().__init__()
        self.parent = parent
        self.emulator_name = emulator_name
        self.emulator = None
        
        # Устанавливаем свойство для идентификации
        self.setProperty("emulator_name", emulator_name)
        
        self.setup_ui()
        self.connect_signals()
    
    def setup_ui(self):
        """Настройка интерфейса (должен быть переопределен)"""
        # Базовый макет - будет переопределен в дочерних классах
        pass
    
    def connect_signals(self):
        """Подключение сигналов"""
        pass
    
    def update_state_display(self):
        """Обновить отображение состояния"""
        if not self.emulator:
            return
        
        # Находим QTextEdit с состоянием
        state_text = self.findChild(QTextEdit)
        if state_text:
            state_info = self.get_state_info()
            state_text.setText(state_info)
    
    def get_state_info(self):
        """Получить информацию о состоянии (должен быть переопределен)"""
        if not self.emulator:
            return "Эмулятор не найден"
        
        state_info = f"Тип: {self.emulator.device_type}\n"
        state_info += f"Состояние: {'Включен' if self.emulator.enabled else 'Выключен'}\n"
        state_info += f"Семистор: {'ПРОБИТ' if getattr(self.emulator, 'triac_shorted', False) else 'Исправен'}\n"
        
        return state_info
    
    def get_quick_commands(self):
        """Получить список быстрых команд (должен быть переопределен)"""
        return []
    
    def create_control_column(self):
        """Создать колонку управления (общий для всех кроме UDP)"""
        col = QVBoxLayout()
        col.setSpacing(5)
        
        # Кнопка включения
        self.enable_btn = QPushButton("▶ Включить")
        self.enable_btn.setCheckable(True)
        self.enable_btn.clicked.connect(lambda checked: self.toggle_emulator(checked))
        col.addWidget(self.enable_btn)
        
        # Настройки
        col.addWidget(QLabel("Настройки:"))
        
        self.cr_check = QCheckBox("Добавить CR (\\r)")
        self.cr_check.setChecked(self.parent.emulator_configs[self.emulator_name]["add_cr"])
        self.cr_check.stateChanged.connect(self.update_cr_setting)
        col.addWidget(self.cr_check)
        
        self.lf_check = QCheckBox("Добавить LF (\\n)")
        self.lf_check.setChecked(self.parent.emulator_configs[self.emulator_name]["add_lf"])
        self.lf_check.stateChanged.connect(self.update_lf_setting)
        col.addWidget(self.lf_check)
        
        col.addStretch()
        return col
    
    def create_settings_column(self):
        """Создать колонку настроек (общий для всех кроме UDP)"""
        col = QVBoxLayout()
        col.setSpacing(5)
        
        # Имитация ошибок
        error_layout = QHBoxLayout()
        error_check = QCheckBox("Имитация ошибок")
        error_check.setChecked(self.parent.emulator_configs[self.emulator_name]["simulate_errors"])
        error_check.stateChanged.connect(self.update_error_setting)
        error_layout.addWidget(error_check)
        
        if self.emulator_name != "UDP_STAB":
            error_spin = QSpinBox()
            error_spin.setRange(0, 100)
            error_spin.setValue(self.parent.emulator_configs[self.emulator_name]["error_rate"])
            error_spin.setSuffix("%")
            error_spin.setMaximumWidth(70)
            error_spin.valueChanged.connect(self.update_error_rate)
            error_layout.addWidget(error_spin)
        
        error_layout.addStretch()
        col.addLayout(error_layout)
        
        # Пробит семистор (добавляем для всех кроме UDP, у которого уже есть)
        if self.emulator_name != "UDP_STAB":
            triac_layout = QHBoxLayout()
            self.triac_check = QCheckBox("Пробит семистор")
            self.triac_check.setToolTip("Режим пробитого симистора/семистора")
            self.triac_check.setChecked(False)
            self.triac_check.stateChanged.connect(self.update_triac_setting)
            triac_layout.addWidget(self.triac_check)
            triac_layout.addStretch()
            col.addLayout(triac_layout)
        
        # Случайная задержка
        delay_layout = QHBoxLayout()
        delay_layout.addWidget(QLabel("Задержка ответа:"))
        
        delay_spin = QSpinBox()
        delay_spin.setRange(0, 10)
        delay_spin.setValue(min(10, self.parent.emulator_configs[self.emulator_name]["random_delay"]))
        delay_spin.setSuffix(" мс")
        delay_spin.setMaximumWidth(100)
        delay_spin.valueChanged.connect(self.update_delay_setting)
        delay_layout.addWidget(delay_spin)
        delay_layout.addStretch()
        col.addLayout(delay_layout)
        
        col.addStretch()
        return col
    
    def create_state_column(self):
        """Создать колонку состояния (общий для всех)"""
        col = QVBoxLayout()
        col.setSpacing(0)
        
        self.state_text = QTextEdit()
        self.state_text.setReadOnly(True)
        self.state_text.setMaximumHeight(200)
        self.state_text.setFont(QFont("Consolas", 9))
        self.state_text.setStyleSheet("""
            QTextEdit {
                background-color: #1a1a2e;
                color: #e6e6e6;
                border: 1px solid #555;
                padding: 2px;
            }
        """)
        col.addWidget(self.state_text, 1)
        
        return col
    
    def create_commands_column(self):
        """Создать колонку быстрых команд (общий для всех кроме UDP)"""
        col = QVBoxLayout()
        col.setSpacing(3)
        
        commands = self.get_quick_commands()
        for command, icon, tooltip in commands:
            btn = QPushButton(icon)
            btn.setToolTip(f"{tooltip}\n{command}" if command else tooltip)
            btn.setProperty("command", command)
            btn.setFixedSize(35, 35)
            btn.setStyleSheet("font-size: 14px;")
            btn.clicked.connect(lambda checked, c=command: self.parent.send_manual_command(c))
            col.addWidget(btn)
        
        col.addStretch()
        return col
    
    def toggle_emulator(self, enabled):
        """Включить/выключить эмулятор (общий для всех)"""
        # Останавливаем текущий эмулятор если он активный
        current_emulator = self.parent.emulator_manager.get_current()
        if current_emulator and current_emulator.enabled and hasattr(current_emulator, 'stop'):
            current_emulator.stop()
        
        # Переключаем на новый эмулятор
        if self.emulator_name in self.parent.emulator_manager.emulators:
            self.parent.emulator_manager.set_current(self.emulator_name)
            self.emulator = self.parent.emulator_manager.get_current()
            
            # Инициализируем чекбокс "Пробит семистор" если он есть
            if hasattr(self, 'triac_check') and self.emulator:
                self.triac_check.setChecked(getattr(self.emulator, 'triac_shorted', False))
            
            # Устанавливаем состояние
            self.emulator.set_enabled(enabled)
            
            # Обновляем текст кнопки
            if self.emulator_name == "UDP_STAB":
                self.enable_btn.setText("▶ Запустить UDP" if not enabled else "⏹ Остановить UDP")
            else:
                self.enable_btn.setText("▶ Включить" if not enabled else "⏹ Выключить")
            
            # Обновляем заголовок окна
            if enabled:
                self.parent.setWindowTitle(f"COM Терминал - {self.emulator_name} (вкл)")
            else:
                self.parent.setWindowTitle("COM Терминал - эмулятор выкл")
            
            from datetime import datetime
            timestamp = get_timestamp()
            self.parent.terminal.append(f"{timestamp}Эмулятор {self.emulator_name} {'включен' if enabled else 'выключен'}")
            
            # Обновляем состояние
            self.update_state_display()
    
    def update_cr_setting(self, state):
        self.parent.emulator_configs[self.emulator_name]["add_cr"] = (state == 2)
    
    def update_lf_setting(self, state):
        self.parent.emulator_configs[self.emulator_name]["add_lf"] = (state == 2)
    
    def update_error_setting(self, state):
        self.parent.emulator_configs[self.emulator_name]["simulate_errors"] = (state == 2)
    
    def update_error_rate(self, value):
        self.parent.emulator_configs[self.emulator_name]["error_rate"] = value
    
    def update_delay_setting(self, value):
        if value > 10:
            value = 10
        self.parent.emulator_configs[self.emulator_name]["random_delay"] = value
    
    def update_triac_setting(self, state):
        """Обновить настройку "Пробит семистор" """
        value = (state == 2)
        
        if self.emulator:
            # Используем метод set_triac_shorted если он есть, иначе устанавливаем напрямую
            if hasattr(self.emulator, 'set_triac_shorted'):
                self.emulator.set_triac_shorted(value)
            else:
                self.emulator.triac_shorted = value
                # Применяем эффект если есть такой метод
                if hasattr(self.emulator, 'apply_triac_effect'):
                    self.emulator.apply_triac_effect()
            
            # Сохраняем в конфиг
            if "simulate_triac" not in self.parent.emulator_configs[self.emulator_name]:
                self.parent.emulator_configs[self.emulator_name]["simulate_triac"] = value
            else:
                self.parent.emulator_configs[self.emulator_name]["simulate_triac"] = value
            
            # Обновляем состояние
            self.update_state_display()
            
            # Логируем
            timestamp = get_timestamp()
            self.parent.terminal.append(
                f"{timestamp}[{self.emulator_name}] Режим 'пробит семистор': {'ВКЛ' if value else 'ВЫКЛ'}"
            )