# Файл: stab_avr_tab.py
"""
Вкладка эмулятора Stab AVR
"""
from PySide6.QtWidgets import *
from .emulator_tab_base import EmulatorTabBase

class StabAVRTab(EmulatorTabBase):
    def __init__(self, parent, emulator_name):
        super().__init__(parent, emulator_name)
        self.emulator = parent.emulator_manager.emulators.get(emulator_name)
        
        # Инициализируем значения префиксов из эмулятора
        self.initialized_prefixes = False
    
    def setup_ui(self):
        """Настройка интерфейса Stab AVR с префиксами"""
        layout = QHBoxLayout(self)
        layout.setContentsMargins(2, 2, 2, 2)
        layout.setSpacing(5)
        
        # Колонка 1: Управление
        col1 = self.create_control_column()
        layout.addLayout(col1, 1)
        
        # Колонка 2: Настройки
        col2 = self.create_settings_column()
        
        # Дополнительные настройки для Stab AVR - префиксы
        prefixes_group = QGroupBox()
        prefixes_layout = QHBoxLayout()
        
        mode_layout = QVBoxLayout()
        mode_layout.setContentsMargins(2, 2, 2, 2)
        mode_layout.addWidget(QLabel("Режим:"))
        self.mode_prefix = QLineEdit("M")
        self.mode_prefix.setMinimumHeight(22)
        self.mode_prefix.setMaximumWidth(50)
        self.mode_prefix.textChanged.connect(self.update_mode_prefix)
        mode_layout.addWidget(self.mode_prefix)
        prefixes_layout.addLayout(mode_layout)
        
        current_layout = QVBoxLayout()
        current_layout.addWidget(QLabel("Текущее:"))
        self.current_prefix = QLineEdit("C")
        self.current_prefix.setMaximumWidth(50)
        self.current_prefix.setMinimumHeight(22)
        self.current_prefix.textChanged.connect(self.update_current_prefix)
        current_layout.addWidget(self.current_prefix)
        prefixes_layout.addLayout(current_layout)
        
        target_layout = QVBoxLayout()
        target_layout.addWidget(QLabel("Целевое:"))
        self.target_prefix = QLineEdit("T")
        self.target_prefix.setMinimumHeight(22)
        self.target_prefix.setMaximumWidth(50)
        self.target_prefix.textChanged.connect(self.update_target_prefix)
        target_layout.addWidget(self.target_prefix)
        prefixes_layout.addLayout(target_layout)
        
        prefixes_group.setLayout(prefixes_layout)
        col2.addWidget(prefixes_group)
        
        # Кнопка применения префиксов
        apply_btn = QPushButton("Применить префиксы")
        apply_btn.clicked.connect(self.apply_prefixes)
        col1.addWidget(apply_btn)
        
        layout.addLayout(col2, 2)
        
        # Колонка 3: Состояние
        col3 = self.create_state_column()
        layout.addLayout(col3, 3)
        
        # Колонка 4: Быстрые команды
        col4 = self.create_commands_column()
        layout.addLayout(col4, 1)
        
        # Инициализируем значения префиксов после создания UI
        self.init_prefixes()
    
    def init_prefixes(self):
        """Инициализировать значения префиксов из эмулятора"""
        if self.emulator:
            # Устанавливаем значения из эмулятора
            self.mode_prefix.setText(getattr(self.emulator, 'mode_prefix', 'M'))
            self.current_prefix.setText(getattr(self.emulator, 'current_prefix', 'C'))
            self.target_prefix.setText(getattr(self.emulator, 'target_prefix', 'T'))
            self.initialized_prefixes = True
    
    def toggle_emulator(self, enabled):
        """Включить/выключить эмулятор с инициализацией префиксов"""
        # Вызываем родительский метод
        super().toggle_emulator(enabled)
        
        # Инициализируем префиксы после включения эмулятора
        if enabled and not self.initialized_prefixes:
            self.init_prefixes()
    
    def get_state_info(self):
        if not self.emulator:
            return "Эмулятор не найден"
        
        state_info = f"Состояние: {'Включен' if self.emulator.enabled else 'Выключен'}\n\n"
        
        state_info += f"Текущая мощность: {getattr(self.emulator, 'current_power', 0)} Вт\n"
        state_info += f"Целевая мощность: {getattr(self.emulator, 'target_power', 0)} Вт\n"
        state_info += f"Режим: {getattr(self.emulator, 'mode', '2')}\n"
        state_info += f"Питание: {'ON' if getattr(self.emulator, 'power_on', False) else 'OFF'}\n\n"
        state_info += f"Семистор: {'ПРОБИТ' if getattr(self.emulator, 'triac_shorted', False) else 'Исправен'}\n"
        state_info += f"Префиксы: {getattr(self.emulator, 'mode_prefix', 'M')}/{getattr(self.emulator, 'current_prefix', 'C')}/{getattr(self.emulator, 'target_prefix', 'T')}\n"
        
        return state_info
    
    def get_quick_commands(self):
        return [
            ("AT+SS?", "📊", "Статус"),
            ("AT+VO?", "⚡", "Текущая мощность"),
            ("AT+VS?", "🎯", "Целевая мощность"),
            ("AT+VS=1500", "1️⃣", "Установить 1500Вт"),
            ("AT+ON=1", "▶", "Включить")
        ]
    
    def update_mode_prefix(self, text):
        """Обновить префикс режима"""
        if self.emulator and hasattr(self.emulator, 'mode_prefix'):
            self.emulator.mode_prefix = text
            self.update_state_display()
    
    def update_current_prefix(self, text):
        """Обновить префикс текущего напряжения"""
        if self.emulator and hasattr(self.emulator, 'current_prefix'):
            self.emulator.current_prefix = text
            self.update_state_display()
    
    def update_target_prefix(self, text):
        """Обновить префикс целевого напряжения"""
        if self.emulator and hasattr(self.emulator, 'target_prefix'):
            self.emulator.target_prefix = text
            self.update_state_display()
    
    def apply_prefixes(self):
        """Применить все префиксы сразу"""
        if self.emulator:
            self.update_mode_prefix(self.mode_prefix.text())
            self.update_current_prefix(self.current_prefix.text())
            self.update_target_prefix(self.target_prefix.text())
            
            # Логируем в терминал
            from utils import get_timestamp
            timestamp = get_timestamp(self.parent)
            self.parent.terminal.append(
                f"{timestamp}[Stab AVR] Префиксы обновлены: "
                f"режим='{self.emulator.mode_prefix}', "
                f"текущее='{self.emulator.current_prefix}', "
                f"целевое='{self.emulator.target_prefix}'"
            )