"""
Вкладка эмулятора UDP стабилизатора (ПОЛНАЯ ВЕРСИЯ)
"""
import json
import os
from datetime import datetime
from PySide6.QtWidgets import *
from PySide6.QtCore import *
from PySide6.QtGui import *
from .emulator_tab_base import EmulatorTabBase
from utils import get_timestamp

class UDPTab(EmulatorTabBase):
    def __init__(self, parent, emulator_name):
        super().__init__(parent, emulator_name)
        self.emulator = parent.emulator_manager.emulators.get(emulator_name)
    
    def setup_ui(self):
        """Настройка интерфейса UDP (специальный макет)"""
        layout = QHBoxLayout(self)
        layout.setContentsMargins(5, 5, 5, 5)
        layout.setSpacing(5)
        
        # ========== КОЛОНКА 1: УПРАВЛЕНИЕ ЭМУЛЯТОРОМ ==========
        col1 = QVBoxLayout()
        col1.setSpacing(8)
        
        # Кнопка включения (высота уменьшена до 25)
        self.enable_btn = QPushButton("▶ Запустить UDP")
        self.enable_btn.setCheckable(True)
        self.enable_btn.setMinimumHeight(25)
        self.enable_btn.setMaximumHeight(25)
        self.enable_btn.clicked.connect(lambda checked: self.toggle_emulator(checked))
        col1.addWidget(self.enable_btn)
        
        # Ручная установка мощности
        power_frame = QFrame()
        power_frame.setFrameShape(QFrame.Box)
        power_frame.setLineWidth(1)
        power_layout = QVBoxLayout(power_frame)
        power_layout.setContentsMargins(5, 5, 5, 5)
        power_layout.setSpacing(5)  # Уменьшаем spacing для равномерного распределения
        
        # Поле для ввода мощности с кнопкой (высота ограничена 22)
        power_input_layout = QHBoxLayout()
        self.power_input = QSpinBox()
        self.power_input.setRange(0, 3000)
        self.power_input.setValue(1000)
        self.power_input.setSuffix(" Вт")
        self.power_input.setMinimumWidth(90)
        self.power_input.setMaximumHeight(22)
        power_input_layout.addWidget(self.power_input)
        
        # Кнопка установки (высота ограничена 22)
        btn_set_power = QPushButton("Установить")
        btn_set_power.setMinimumHeight(22)
        btn_set_power.setMaximumHeight(22)
        btn_set_power.clicked.connect(self.set_power_from_input)
        power_input_layout.addWidget(btn_set_power)
        power_input_layout.addStretch()
        power_layout.addLayout(power_input_layout)
        
        # Кнопки ручного управления (высота ограничена 22)
        power_buttons_layout = QGridLayout()
        power_buttons_layout.setVerticalSpacing(4)
        power_buttons_layout.setHorizontalSpacing(4)
        
        # Ряд 1 (высота кнопок 22)
        btn_500 = QPushButton("500 Вт")
        btn_500.setMinimumHeight(22)
        btn_500.setMaximumHeight(22)
        btn_500.clicked.connect(lambda: self.set_power(500))
        power_buttons_layout.addWidget(btn_500, 0, 0)
        
        btn_1000 = QPushButton("1000 Вт")
        btn_1000.setMinimumHeight(22)
        btn_1000.setMaximumHeight(22)
        btn_1000.clicked.connect(lambda: self.set_power(1000))
        power_buttons_layout.addWidget(btn_1000, 0, 1)
        
        # Ряд 2 (высота кнопок 22)
        btn_1500 = QPushButton("1500 Вт")
        btn_1500.setMinimumHeight(22)
        btn_1500.setMaximumHeight(22)
        btn_1500.clicked.connect(lambda: self.set_power(1500))
        power_buttons_layout.addWidget(btn_1500, 1, 0)
        
        btn_2000 = QPushButton("2000 Вт")
        btn_2000.setMinimumHeight(22)
        btn_2000.setMaximumHeight(22)
        btn_2000.clicked.connect(lambda: self.set_power(2000))
        power_buttons_layout.addWidget(btn_2000, 1, 1)
        
        # Ряд 3 (высота кнопок 22)
        self.btn_boost = QPushButton("Разгон")
        self.btn_boost.setMinimumHeight(22)
        self.btn_boost.setMaximumHeight(22)
        self.btn_boost.clicked.connect(self.set_boost)
        power_buttons_layout.addWidget(self.btn_boost, 2, 0)
        
        self.btn_off = QPushButton("Выключить")
        self.btn_off.setMinimumHeight(22)
        self.btn_off.setMaximumHeight(22)
        self.btn_off.clicked.connect(self.set_off)
        power_buttons_layout.addWidget(self.btn_off, 2, 1)
        
        power_layout.addLayout(power_buttons_layout)
        col1.addWidget(power_frame)
        
        col1.addStretch()
        layout.addLayout(col1, 2)
        
        # ========== КОЛОНКА 2: НАСТРОЙКИ ==========
        col2 = QVBoxLayout()
        col2.setSpacing(2)
        
        # Имитация ошибок
        error_layout = QHBoxLayout()
        self.error_check = QCheckBox("Имитация ошибок")
        self.error_check.setChecked(self.parent.emulator_configs[self.emulator_name]["simulate_errors"])
        self.error_check.stateChanged.connect(self.update_error_setting)
        error_layout.addWidget(self.error_check)
        error_layout.addStretch()
        col2.addLayout(error_layout)
        
        # Галка "Пробит семистор"
        triac_layout = QHBoxLayout()
        self.triac_check = QCheckBox("Пробит семистор")
        self.triac_check.setToolTip("Текущая мощность всегда 3000W при включенном нагреве")
        self.triac_check.setChecked(self.parent.emulator_configs[self.emulator_name].get("simulate_triac", False))
        self.triac_check.stateChanged.connect(self.update_triac_setting)
        triac_layout.addWidget(self.triac_check)
        triac_layout.addStretch()
        col2.addLayout(triac_layout)
        
        # Случайная задержка
        delay_layout = QHBoxLayout()
        delay_layout.addWidget(QLabel("Задержка ответа:"))
        
        self.delay_spin = QSpinBox()
        self.delay_spin.setRange(0, 10)
        self.delay_spin.setValue(min(10, self.parent.emulator_configs[self.emulator_name]["random_delay"]))
        self.delay_spin.setSuffix(" мс")
        self.delay_spin.setMaximumWidth(80)
        self.delay_spin.valueChanged.connect(self.update_delay_setting)
        delay_layout.addWidget(self.delay_spin)
        delay_layout.addStretch()
        col2.addLayout(delay_layout)
        
        # Настройки UDP
        udp_frame = QFrame()
        udp_frame.setFrameShape(QFrame.Box)
        udp_frame.setLineWidth(1)
        udp_layout = QVBoxLayout(udp_frame)
        udp_layout.setContentsMargins(5, 8, 5, 8)
        
        # IP и порт
        ip_port_layout = QHBoxLayout()
        ip_port_layout.addWidget(QLabel("IP:"))
        
        initial_ip = self.emulator.ip_address if self.emulator and hasattr(self.emulator, 'ip_address') else "192.168.1.100"
        
        self.ip_edit = QLineEdit(initial_ip)
        self.ip_edit.setMinimumWidth(100)
        self.ip_edit.textChanged.connect(self.update_udp_ip)
        ip_port_layout.addWidget(self.ip_edit)
        
        ip_port_layout.addWidget(QLabel(":"))
        self.port_spin = QSpinBox()
        self.port_spin.setRange(1024, 65535)
        self.port_spin.setValue(12345)
        self.port_spin.setMaximumWidth(80)
        self.port_spin.valueChanged.connect(self.update_udp_port)
        ip_port_layout.addWidget(self.port_spin)
        ip_port_layout.addStretch()
        udp_layout.addLayout(ip_port_layout)
        
        # Интервал
        interval_layout = QHBoxLayout()
        interval_layout.addWidget(QLabel("Интервал:"))
        self.interval_spin = QSpinBox()
        self.interval_spin.setRange(100, 10000)
        self.interval_spin.setValue(1000)
        self.interval_spin.setSuffix(" мс")
        self.interval_spin.setMaximumWidth(80)
        self.interval_spin.valueChanged.connect(self.update_udp_interval)
        interval_layout.addWidget(self.interval_spin)
        interval_layout.addStretch()
        udp_layout.addLayout(interval_layout)
        
        col2.addWidget(udp_frame)
        
        col2.addStretch()
        layout.addLayout(col2, 2)
        
        # ========== КОЛОНКА 3: СОСТОЯНИЕ ==========
        col3 = self.create_state_column()
        layout.addLayout(col3, 3)
    
    def get_state_info(self):
        if not self.emulator:
            return "Эмулятор не найден"
        
        state_info = f"Тип: {self.emulator.device_type}\n"
        state_info += f"Состояние: {'Включен' if self.emulator.enabled else 'Выключен'}\n\n"
        
        state_info += f"Текущая мощность: {int(getattr(self.emulator, 'current_power', 0))} Вт\n"
        state_info += f"Целевая мощность: {int(getattr(self.emulator, 'target_power', 0))} Вт\n"
        state_info += f"Нагрев: {'Вкл' if getattr(self.emulator, 'heat_enabled', False) else 'Выкл'}\n"
        state_info += f"Режим разгона: {'Вкл' if getattr(self.emulator, 'boost_mode', False) else 'Выкл'}\n"
        state_info += f"Режим ошибки: {'ВКЛ (0x04)' if getattr(self.emulator, 'error_mode', False) else 'Выкл'}\n"
        state_info += f"Семистор пробит: {'Да' if getattr(self.emulator, 'triac_shorted', False) else 'Нет'}\n"
        
        # Показать фазу стабилизации если есть
        if hasattr(self.emulator, 'stabilization_phase'):
            phases = {
                "idle": "ожидание",
                "rapid": "быстрая стабилизация", 
                "slow": "медленная стабилизация",
                "stable": "стабилизировано"
            }
            state_info += f"Стабилизация: {phases.get(self.emulator.stabilization_phase, 'неизвестно')}\n"
        
        state_info += f"IP адрес: {getattr(self.emulator, 'ip_address', 'N/A')}\n"
        state_info += f"UDP порт: {getattr(self.emulator, 'udp_port', 0)}\n"
        state_info += f"UDP рассылка: {'Активна' if getattr(self.emulator, 'running', False) else 'Остановлена'}\n"
        
        return state_info
    
    def set_power_from_input(self):
        power = self.power_input.value()
        self.set_power(power)
    
    def set_power(self, power):
        if not self.emulator:
            return
        
        timestamp = get_timestamp()
        
        # Если включен режим ошибки - сбрасываем его
        if self.emulator.error_mode:
            self.emulator.error_mode = False
            self.emulator.error_start_time = 0
            self.error_check.setChecked(False)
        
        # Устанавливаем мощность
        self.emulator.target_power = power
        self.emulator.heat_enabled = True
        self.emulator.boost_mode = False
        
        # Сбрасываем стабилизацию
        if hasattr(self.emulator, 'stabilization_phase'):
            self.emulator.stabilization_phase = "idle"
            self.emulator.stabilization_start_time = None
        
        # Если семистор не пробит, начинаем стабилизацию
        if not self.emulator.triac_shorted:
            self.emulator.current_power = int(power * 0.8)
            self.emulator.stabilization_phase = "rapid"
            self.emulator.stabilization_start_time = datetime.now().timestamp()
            self.emulator.rapid_target = int(power * 1.05)
        else:
            # Если семистор пробит, текущая мощность 3000
            self.emulator.current_power = 3000
        
        # Логируем
        self.parent.terminal.append(f"{timestamp}[UDP] Ручная установка мощности: {power} Вт")
        
        # Обновляем состояние
        self.update_state_display()
    
    def set_boost(self):
        if not self.emulator:
            return
        
        timestamp = get_timestamp()
        
        # Включаем разгон
        if not self.emulator.boost_mode:
            # Сохраняем текущее состояние
            if hasattr(self.emulator, 'pre_boost_state'):
                self.emulator.pre_boost_state['target_power'] = self.emulator.target_power
                self.emulator.pre_boost_state['heat_enabled'] = self.emulator.heat_enabled
            
            self.emulator.boost_mode = True
            self.emulator.heat_enabled = True
            self.emulator.error_mode = False
            self.error_check.setChecked(False)
            
            # Устанавливаем максимальную мощность
            self.emulator.current_power = 3000
            self.emulator.target_power = 3000
            
            # Сбрасываем стабилизацию
            if hasattr(self.emulator, 'stabilization_phase'):
                self.emulator.stabilization_phase = "idle"
                self.emulator.stabilization_start_time = None
            
            self.parent.terminal.append(f"{timestamp}[UDP] Ручное включение режима разгона")
            
            # Обновляем состояние
            self.update_state_display()
    
    def set_off(self):
        if not self.emulator:
            return
        
        timestamp = get_timestamp()
        
        # Выключаем нагрев и разгон, сбрасываем ошибку
        self.emulator.heat_enabled = False
        self.emulator.boost_mode = False
        self.emulator.error_mode = False
        self.emulator.error_start_time = 0
        self.error_check.setChecked(False)
        self.emulator.current_power = 0
        
        # Сбрасываем стабилизацию
        if hasattr(self.emulator, 'stabilization_phase'):
            self.emulator.stabilization_phase = "idle"
            self.emulator.stabilization_start_time = None
        
        self.parent.terminal.append(f"{timestamp}[UDP] Ручное выключение нагрева и разгона")
        
        # Обновляем состояние
        self.update_state_display()
    
    def update_error_setting(self, state):
        value = (state == 2)
        self.parent.emulator_configs[self.emulator_name]["simulate_errors"] = value
        
        if self.emulator:
            if value:
                # Включаем режим ошибки
                self.emulator.error_mode = True
                self.emulator.error_start_time = datetime.now().timestamp()
                self.emulator.heat_enabled = False
                self.emulator.current_power = 0
                if hasattr(self.emulator, 'stabilization_phase'):
                    self.emulator.stabilization_phase = "idle"
            else:
                # Выключаем режим ошибки
                self.emulator.error_mode = False
                self.emulator.error_start_time = 0
            
            self.update_state_display()
    
    def update_triac_setting(self, state):
        value = (state == 2)
        self.parent.emulator_configs[self.emulator_name]["simulate_triac"] = value
        
        if self.emulator:
            if hasattr(self.emulator, 'triac_shorted'):
                # Если включен режим ошибки - не меняем настройки семистора
                if self.emulator.error_mode:
                    return
                
                self.emulator.triac_shorted = value
                self.update_state_display()
    
    def update_delay_setting(self, value):
        if value > 10:
            value = 10
            self.delay_spin.setValue(10)
        self.parent.emulator_configs[self.emulator_name]["random_delay"] = value
    
    def update_udp_ip(self, text):
        if self.emulator:
            self.emulator.ip_address = text
    
    def update_udp_port(self, value):
        if self.emulator:
            self.emulator.udp_port = value
    
    def update_udp_interval(self, value):
        if self.emulator:
            self.emulator.broadcast_interval = value