"""
Главное окно терминала
"""
import random
import time
import re
from datetime import datetime
from PySide6.QtWidgets import *
from PySide6.QtCore import *
from PySide6.QtGui import *

from serial_manager import SerialManager
from emulators.manager import EmulatorManager
from ui_components import ControlPanel, SendPanel
from hc12_tab import create_hc12_tab, parse_hc12_response
from tabs import create_all_emulator_tabs
from config_manager import ConfigManager
from utils import get_timestamp

class FinalTerminal(QMainWindow):
    def __init__(self):
        super().__init__()
        
        # Менеджер конфигурации
        self.config = ConfigManager()
        
        # Менеджеры
        self.serial_manager = SerialManager(self)
        self.emulator_manager = EmulatorManager(self)
        
        # UI элементы
        self.terminal = None
        self.tab_widget = None
        self.send_panel = None
        
        # Настройки эмуляторов
        self.emulator_configs = {
            "RMVK": {"add_cr": True, "add_lf": True, "simulate_errors": False, "error_rate": 10, "random_delay": 0},
            "KVIC9600": {"add_cr": True, "add_lf": True, "simulate_errors": False, "error_rate": 10, "random_delay": 0},
            "STAB_AVR": {"add_cr": True, "add_lf": True, "simulate_errors": False, "error_rate": 10, "random_delay": 0},
            "UNI_PROTOCOL": {"add_cr": True, "add_lf": True, "simulate_errors": False, "error_rate": 10, "random_delay": 0},
            "MODBUS_RTU": {"add_cr": True, "add_lf": True, "simulate_errors": False, "error_rate": 10, "random_delay": 0},
            "UDP_STAB": {"add_cr": False, "add_lf": False, "simulate_errors": False, "error_rate": 0, "random_delay": 0, "simulate_triac": False},
        }
        
        # Загружаем настройку timestamp
        self.show_timestamp = self.config.get_timestamp_setting()
        if hasattr(self, 'control_panel') and hasattr(self.control_panel, 'timestamp_btn'):
            self.control_panel.timestamp_btn.setChecked(self.show_timestamp)
            self.control_panel.show_timestamp = self.show_timestamp
        
        self.setup_ui()
        self.setup_context_menu()
        self.setup_timers()
        
        print("[INIT] Терминал инициализирован")
    
    def setup_ui(self):
        self.setWindowTitle("COM Терминал + Эмуляторы стабилизаторов + HC-12")
        
        # Загружаем настройки окна
        window_settings = self.config.get_window_settings()
        self.setGeometry(
            window_settings['x'],
            window_settings['y'],
            window_settings['width'],
            window_settings['height']
        )
        
        if window_settings['maximized']:
            self.showMaximized()
        
        # Центральный виджет
        central = QWidget()
        self.setCentralWidget(central)
        main_layout = QVBoxLayout(central)
        main_layout.setContentsMargins(3, 3, 3, 3)
        main_layout.setSpacing(3)
        
        # Панель управления
        self.control_panel = ControlPanel(self)
        main_layout.addWidget(self.control_panel)
        
        # Терминал
        self.setup_terminal()
        main_layout.addWidget(self.terminal, 1)
        
        # Панель отправки
        self.send_panel = SendPanel(self)
        main_layout.addWidget(self.send_panel)
        
        # Вкладки
        self.setup_tabs()
        self.tab_widget.setMaximumHeight(150)  # Фиксируем максимальную высоту вкладок
        main_layout.addWidget(self.tab_widget)
        
        print("[UI] Интерфейс создан")
    
    def setup_terminal(self):
        """Настройка терминала с загрузкой цветов из конфига"""
        self.terminal = QTextEdit()
        self.terminal.setReadOnly(True)
        
        # Загружаем настройки терминала
        terminal_settings = self.config.get_terminal_colors()
        font = QFont(
            terminal_settings['font_family'],
            int(terminal_settings['font_size'])
        )
        self.terminal.setFont(font)
        
        self.update_terminal_colors(
            terminal_settings['bg_color'],
            terminal_settings['text_color']
        )
    
    def setup_context_menu(self):
        """Настройка контекстного меню терминала"""
        self.terminal.setContextMenuPolicy(Qt.CustomContextMenu)
        self.terminal.customContextMenuRequested.connect(self.show_terminal_context_menu)
    
    def show_terminal_context_menu(self, position):
        """Показать контекстное меню терминала"""
        menu = self.terminal.createStandardContextMenu()
        
        # Добавляем разделитель
        menu.addSeparator()
        
        # Цвет фона
        bg_menu = QMenu("Цвет фона", self)
        bg_colors = {
            "Темно-синий": "#0a1929",
            "Черный": "#000000",
            "Темно-серый": "#1a1a1a",
            "Темно-зеленый": "#002200",
            "Темно-фиолетовый": "#1a0029"
        }
        
        for name, color in bg_colors.items():
            action = bg_menu.addAction(name)
            action.triggered.connect(lambda checked, c=color: self.set_terminal_bg_color(c))
        
        # Цвет текста
        text_menu = QMenu("Цвет текста", self)
        text_colors = {
            "Светло-голубой": "#e6f7ff",
            "Белый": "#ffffff",
            "Зеленый": "#00ff00",
            "Желтый": "#ffff00",
            "Голубой": "#00ffff"
        }
        
        for name, color in text_colors.items():
            action = text_menu.addAction(name)
            action.triggered.connect(lambda checked, c=color: self.set_terminal_text_color(c))
        
        # Размер шрифта
        font_menu = QMenu("Размер шрифта", self)
        font_sizes = ["8", "9", "10", "11", "12", "14", "16"]
        
        for size in font_sizes:
            action = font_menu.addAction(size)
            action.triggered.connect(lambda checked, s=size: self.set_terminal_font_size(int(s)))
        
        menu.addMenu(bg_menu)
        menu.addMenu(text_menu)
        menu.addMenu(font_menu)
        menu.addSeparator()
        
        # Очистить терминал
        clear_action = menu.addAction("Очистить терминал")
        clear_action.triggered.connect(self.terminal.clear)
        
        menu.exec(self.terminal.viewport().mapToGlobal(position))
    
    def update_terminal_colors(self, bg_color=None, text_color=None):
        """Надежное обновление цветов терминала"""
        # Получаем текущие настройки из конфига, если цвета не указаны
        terminal_settings = self.config.get_terminal_colors()
        bg_color = bg_color or terminal_settings['bg_color']
        text_color = text_color or terminal_settings['text_color']
        
        # Просто задаем стиль напрямую
        style = f"""
        QTextEdit {{
            background-color: {bg_color};
            color: {text_color};
            border: 1px solid #555;
        }}
        """
        self.terminal.setStyleSheet(style)
        
        # Также обновляем палитру
        palette = self.terminal.palette()
        palette.setColor(QPalette.Base, QColor(bg_color))
        palette.setColor(QPalette.Text, QColor(text_color))
        self.terminal.setPalette(palette)
    
    def set_terminal_bg_color(self, color):
        """Установить цвет фона терминала"""
        terminal_settings = self.config.get_terminal_colors()
        
        # Обновляем только фон
        self.update_terminal_colors(
            bg_color=color,
            text_color=terminal_settings['text_color']
        )
        
        # Сохраняем в конфиг
        self.config.save_terminal_colors(
            color,
            terminal_settings['text_color'],
            terminal_settings['font_family'],
            terminal_settings['font_size']
        )
    
    def set_terminal_text_color(self, color):
        """Установить цвет текста терминала"""
        terminal_settings = self.config.get_terminal_colors()
        
        # Обновляем только цвет текста
        self.update_terminal_colors(
            bg_color=terminal_settings['bg_color'],
            text_color=color
        )
        
        # Сохраняем в конфиг
        self.config.save_terminal_colors(
            terminal_settings['bg_color'],
            color,
            terminal_settings['font_family'],
            terminal_settings['font_size']
        )
    
    def set_terminal_font_size(self, size):
        """Установить размер шрифта терминала"""
        font = self.terminal.font()
        font.setPointSize(size)
        self.terminal.setFont(font)
        
        # Сохраняем в конфиг
        terminal_settings = self.config.get_terminal_colors()
        self.config.save_terminal_colors(
            terminal_settings['bg_color'],
            terminal_settings['text_color'],
            terminal_settings['font_family'],
            size
        )
    
    def setup_tabs(self):
        self.tab_widget = QTabWidget()
        
        # Вкладка Пресеты (первая)
        from presets_tab import PresetsTab
        presets_tab = PresetsTab(self)
        self.tab_widget.addTab(presets_tab, "📋 Пресеты")
        
        # Создаем вкладки эмуляторов
        create_all_emulator_tabs(self.tab_widget, self)
        
        # Вкладка HC-12
        hc12_tab = create_hc12_tab(self)
        self.tab_widget.addTab(hc12_tab, "📡 HC-12")
        
        # Загружаем настройки HC-12
        hc12_settings = self.config.get_hc12_settings()
        if hasattr(self, 'hc12_channel'):
            index = self.hc12_channel.findText(hc12_settings['channel'])
            if index >= 0:
                self.hc12_channel.setCurrentIndex(index)
        
        if hasattr(self, 'hc12_baud'):
            index = self.hc12_baud.findText(hc12_settings['baud'])
            if index >= 0:
                self.hc12_baud.setCurrentIndex(index)
        
        if hasattr(self, 'hc12_power'):
            for i in range(self.hc12_power.count()):
                if hc12_settings['power'] in self.hc12_power.itemText(i):
                    self.hc12_power.setCurrentIndex(i)
                    break
        
        if hasattr(self, 'hc12_mode'):
            for i in range(self.hc12_mode.count()):
                if hc12_settings['mode'] in self.hc12_mode.itemText(i):
                    self.hc12_mode.setCurrentIndex(i)
                    break
        
        # Подключение сигналов
        self.tab_widget.currentChanged.connect(self.on_tab_changed)
    
    def setup_timers(self):
        """Настройка таймеров"""
        self.emulator_timer = QTimer()
        self.emulator_timer.timeout.connect(self.update_emulators)
        self.emulator_timer.start(100)  # 10 раз в секунду
    
    def update_emulators(self):
        """Обновление эмуляторов"""
        self.emulator_manager.periodic_update()
        
        # Обновляем отображение состояния
        for i in range(self.tab_widget.count()):
            tab = self.tab_widget.widget(i)
            emulator_name = getattr(tab, 'emulator_name', None)
            if emulator_name:
                self.update_emulator_state_display(emulator_name)
    
    def on_tab_changed(self, index):
        """Обработка изменения вкладки"""
        # Скрываем все вкладки кроме текущей
        for i in range(self.tab_widget.count()):
            widget = self.tab_widget.widget(i)
            if i == index:
                widget.show()
            else:
                widget.hide()
        
        # Фиксируем высоту вкладки
        self.adjust_tab_height(index)
    
    def adjust_tab_height(self, index):
        """Подстроить высоту панели вкладки под содержимое"""
        widget = self.tab_widget.widget(index)
        if widget:
            # Получаем оптимальную высоту содержимого
            height = widget.sizeHint().height()
            # Устанавливаем фиксированную высоту (немного больше оптимальной)
            self.tab_widget.setMaximumHeight(min(180, height + 10))
    
    def resizeEvent(self, event):
        """Обработка изменения размера окна"""
        super().resizeEvent(event)
        # Высота вкладок не меняется при изменении размера окна
        # Они остаются фиксированной высоты
    
    def update_emulator_state_display(self, emulator_name):
        """Обновить отображение состояния эмулятора"""
        # Делегируем обновление соответствующей вкладке
        for i in range(self.tab_widget.count()):
            tab = self.tab_widget.widget(i)
            if getattr(tab, 'emulator_name', None) == emulator_name:
                if hasattr(tab, 'update_state_display'):
                    tab.update_state_display()
                break
    
    # ========== МЕТОДЫ ДЛЯ ОБРАБОТКИ КОМАНД ==========
    
    def send_manual_command(self, command: str):
        """Отправить команду из ручного управления"""
        if not command:
            return
        
        # Логируем отправку
        timestamp = get_timestamp(self)
        self.terminal.append(f"{timestamp}TX (manual): {command}")
        
        # Обрабатываем команду как входящие данные
        self.process_received_data(command)
        
        # Если нужно, также отправляем через SerialManager
        if self.serial_manager.is_connected():
            emulator_name = self.emulator_manager.current_emulator
            config = self.emulator_configs.get(emulator_name, {})
            
            command_to_send = command
            if config.get("add_cr", True):
                command_to_send += "\r"
            if config.get("add_lf", True):
                command_to_send += "\n"
            
            self.serial_manager.send_text(command_to_send)
    
    def send_hc12_command(self, command: str):
        """Отправить команду HC-12"""
        # Для HC-12 всегда \r\n
        hc12_command = command + "\r\n"
        
        if self.serial_manager.is_connected():
            if self.serial_manager.send_text(hc12_command):
                timestamp = self.get_timestamp()
                self.terminal.append(f"{timestamp}HC-12 >> {command}")
                
                # Сохраняем настройки HC-12
                if hasattr(self, 'hc12_channel') and hasattr(self, 'hc12_baud') and \
                   hasattr(self, 'hc12_power') and hasattr(self, 'hc12_mode'):
                    self.config.save_hc12_settings(
                        self.hc12_channel.currentText(),
                        self.hc12_baud.currentText(),
                        self.hc12_power.currentText(),
                        self.hc12_mode.currentText()
                    )
    
    def process_received_data(self, data: str):
        """Обработка полученных текстовых данных"""
        if not data:
            return
        
        # Логируем
        timestamp = get_timestamp()
        self.terminal.append(f"{timestamp}RX: {data}")
        
        # Автопрокрутка
        if self.send_panel and self.send_panel.autoscroll_check.isChecked():
            cursor = self.terminal.textCursor()
            cursor.movePosition(QTextCursor.End)
            self.terminal.setTextCursor(cursor)
        
        # Обработка эмулятором
        emulator = self.emulator_manager.get_current()
        if emulator and emulator.enabled:
            emulator_name = emulator.device_type
            config = self.emulator_configs.get(emulator_name, {})
            
            # Имитация случайной задержки
            if config.get("random_delay", 0) > 0:
                delay = random.randint(0, config["random_delay"])
                time.sleep(delay / 1000.0)
            
            # Имитация ошибок
            if config.get("simulate_errors", False) and random.randint(0, 100) < config.get("error_rate", 10):
                self.terminal.append(f"{timestamp}[Эмулятор] ОШИБКА (имитация)")
                return
            
            response = self.emulator_manager.process_command(data, is_binary=False)
            if response:
                # Добавляем окончание строки если нужно
                response_to_send = response
                if config.get("add_cr", True):
                    response_to_send += "\r"
                if config.get("add_lf", True):
                    response_to_send += "\n"
                
                # Отправляем ответ
                if self.serial_manager.send_text(response_to_send):
                    self.terminal.append(f"{timestamp}Эмулятор >> {response}")
                    
                    # Обновляем состояние
                    self.update_emulator_state_display(emulator_name)
        
        # Обработка HC-12
        if any(x in data for x in ["OK+", "AT+RX", "FU", "AT+P"]):
            if hasattr(self, 'hc12_log'):
                self.hc12_log.append(data)
            
            # Парсинг ответа
            parse_hc12_response(data, self)
    
    def process_received_binary(self, data: bytes):
        """Обработка полученных бинарных данных (для Modbus)"""
        if not data:
            return
        
        # Логируем
        timestamp = get_timestamp()
        hex_str = data.hex().upper()
        self.terminal.append(f"{timestamp}RX (HEX): {hex_str}")
        
        # Обработка эмулятором Modbus
        emulator = self.emulator_manager.get_current()
        if (emulator and emulator.enabled and
            emulator.device_type == "MODBUS_RTU"):
            
            emulator_name = "MODBUS_RTU"
            config = self.emulator_configs.get(emulator_name, {})
            
            # Имитация случайной задержки
            if config.get("random_delay", 0) > 0:
                delay = random.randint(0, config["random_delay"])
                time.sleep(delay / 1000.0)
            
            # Имитация ошибок
            if config.get("simulate_errors", False) and random.randint(0, 100) < config.get("error_rate", 10):
                self.terminal.append(f"{timestamp}[Modbus] ОШИБКА (имитация)")
                return
            
            response = self.emulator_manager.process_command(data, is_binary=True)
            if response:
                # Отправляем ответ
                if self.serial_manager.send_binary(response):
                    hex_response = response.hex().upper()
                    self.terminal.append(f"{timestamp}Modbus >> (HEX): {hex_response}")
                    
                    # Обновляем состояние
                    self.update_emulator_state_display("MODBUS_RTU")
    
    def get_timestamp(self):
        """Получить текущее время в формате строки с миллисекундами"""
        # Используем функцию из utils
        from utils import get_timestamp
        return get_timestamp(self)  # Передаем self как parent
        
    def closeEvent(self, event):
        """Обработка закрытия окна"""
        # Сохраняем настройки окна
        self.config.save_window_settings(self)
        
        # Сохраняем настройки HC-12
        if hasattr(self, 'hc12_channel') and hasattr(self, 'hc12_baud') and \
            hasattr(self, 'hc12_power') and hasattr(self, 'hc12_mode'):
            self.config.save_hc12_settings(
                self.hc12_channel.currentText(),
                self.hc12_baud.currentText(),
                self.hc12_power.currentText(),
                self.hc12_mode.currentText()
            )
        
        # Останавливаем все эмуляторы
        for emulator in self.emulator_manager.emulators.values():
            if hasattr(emulator, 'stop'):
                emulator.stop()
        
        # Останавливаем COM порт
        self.serial_manager.disconnect()
        
        # Останавливаем таймеры
        if hasattr(self, 'emulator_timer') and self.emulator_timer.isActive():
            self.emulator_timer.stop()
        
        event.accept()