"""
UI компоненты для терминала
"""
from datetime import datetime
from PySide6.QtWidgets import *
from PySide6.QtCore import *
from PySide6.QtGui import *
from utils import get_timestamp

class ControlPanel(QFrame):
    """Панель управления"""
    def __init__(self, parent=None):
        super().__init__(parent)
        self.parent = parent
        self.show_timestamp = True  # По умолчанию включен
        self.setup_ui()
    
    def setup_ui(self):
        self.setMaximumHeight(40)
        layout = QHBoxLayout(self)
        layout.setContentsMargins(5, 5, 5, 5)
        layout.setSpacing(5)
        
        # Кнопка Timestamp
        self.timestamp_btn = QPushButton("🕐")
        self.timestamp_btn.setCheckable(True)
        self.timestamp_btn.setChecked(True)
        self.timestamp_btn.setToolTip("Включить/выключить Timestamp с миллисекундами")
        self.timestamp_btn.setFixedSize(30, 30)
        self.timestamp_btn.clicked.connect(self.toggle_timestamp)
        layout.addWidget(self.timestamp_btn)
        
        # Кнопка обновления портов
        self.refresh_btn = QPushButton("🔄")
        self.refresh_btn.setToolTip("Обновить список портов")
        self.refresh_btn.setFixedSize(30, 30)
        self.refresh_btn.clicked.connect(self.refresh_ports)
        layout.addWidget(self.refresh_btn)
        
        # Порт
        layout.addWidget(QLabel("Порт:"))
        self.port_combo = QComboBox()
        self.port_combo.setMinimumWidth(200)
        layout.addWidget(self.port_combo)
        
        # Скорость
        layout.addWidget(QLabel("Скорость:"))
        self.baud_combo = QComboBox()
        self.baud_combo.addItems(["9600", "38400", "115200", "57600", "19200", "2400", "4800", "1200"])
        self.baud_combo.setCurrentText("9600")
        self.baud_combo.setMinimumWidth(100)
        layout.addWidget(self.baud_combo)
        
        # Кнопки
        self.connect_btn = QPushButton("📌")
        self.connect_btn.setToolTip("Подключить/отключить")
        self.connect_btn.setFixedSize(30, 30)
        self.connect_btn.clicked.connect(self.toggle_connection)
        layout.addWidget(self.connect_btn)
        
        self.clear_btn = QPushButton("🗑")
        self.clear_btn.setToolTip("Очистить терминал")
        self.clear_btn.setFixedSize(30, 30)
        self.clear_btn.clicked.connect(self.clear_terminal)
        layout.addWidget(self.clear_btn)
        
        self.save_btn = QPushButton("💾")
        self.save_btn.setToolTip("Сохранить лог")
        self.save_btn.setFixedSize(30, 30)
        self.save_btn.clicked.connect(self.save_log)
        layout.addWidget(self.save_btn)
        
        layout.addStretch()
        
        # Обновляем список портов
        self.refresh_ports()
    
    def toggle_timestamp(self, checked):
        """Включить/выключить Timestamp"""
        self.show_timestamp = checked
        if self.parent and hasattr(self.parent, 'config'):
            self.parent.config.save_timestamp_setting(checked)
        
        if checked:
            self.timestamp_btn.setToolTip("Timestamp включен (с миллисекундами)")
            self.timestamp_btn.setStyleSheet("background-color: #4a9eff;")
        else:
            self.timestamp_btn.setToolTip("Timestamp выключен")
            self.timestamp_btn.setStyleSheet("")
    
    def refresh_ports(self):
        """Обновить список портов"""
        self.port_combo.clear()
        ports = self.parent.serial_manager.scan_ports()
        
        for port_device, port_description in ports:
            self.port_combo.addItem(f"{port_device} - {port_description}", port_device)
        
        if not ports:
            self.port_combo.addItem("Нет доступных портов")
    
    def toggle_connection(self):
        """Подключение/отключение"""
        if self.parent.serial_manager.is_connected():
            self.parent.serial_manager.disconnect()
            self.connect_btn.setText("📌")
        else:
            port_name = self.port_combo.currentData()
            baud_rate = int(self.baud_combo.currentText())
            if port_name and port_name != "Нет доступных портов":
                if self.parent.serial_manager.connect(port_name, baud_rate):
                    self.connect_btn.setText("🔌")
    
    def clear_terminal(self):
        """Очистить терминал"""
        if self.parent.terminal:
            self.parent.terminal.clear()
            from datetime import datetime
            self.parent.terminal.append(f"[{datetime.now().strftime('%H:%M:%S')}] Терминал очищен")
    
    def save_log(self):
        """Сохранить лог"""
        from PySide6.QtWidgets import QFileDialog
        
        filename, _ = QFileDialog.getSaveFileName(
            self, "Сохранить лог", "", "Текстовые файлы (*.txt);;Все файлы (*)"
        )
        
        if filename and self.parent.terminal:
            try:
                with open(filename, 'w', encoding='utf-8') as f:
                    f.write(self.parent.terminal.toPlainText())
                self.parent.terminal.append(f"[{datetime.now().strftime('%H:%M:%S')}] Лог сохранен в {filename}")
            except Exception as e:
                self.parent.terminal.append(f"[{datetime.now().strftime('%H:%M:%S')}] ОШИБКА: Не удалось сохранить: {str(e)}")

class SendPanel(QFrame):
    """Панель отправки"""
    def __init__(self, parent=None):
        super().__init__(parent)
        self.parent = parent
        self.setup_ui()
    
    def setup_ui(self):
        self.setMaximumHeight(40)
        layout = QHBoxLayout(self)
        layout.setContentsMargins(5, 5, 5, 5)
        
        self.input_field = QLineEdit()
        self.input_field.setPlaceholderText("Введите команду и нажмите Enter...")
        self.input_field.returnPressed.connect(self.send_command)
        layout.addWidget(self.input_field)
        
        self.send_btn = QPushButton("📤")
        self.send_btn.setToolTip("Отправить команду")
        self.send_btn.setFixedSize(30, 30)
        self.send_btn.clicked.connect(self.send_command)
        layout.addWidget(self.send_btn)
        
        # Чекбокс верхнего регистра
        self.upper_check = QCheckBox("ВВЕРХ")
        self.upper_check.setToolTip("Исправить регистр (перевести в верхний регистр)")
        layout.addWidget(self.upper_check)
        
        # Чекбокс автопрокрутки
        self.autoscroll_check = QCheckBox("Автопрокрутка")
        self.autoscroll_check.setChecked(True)
        layout.addWidget(self.autoscroll_check)
        
        # Выбор окончания строки
        self.eol_combo = QComboBox()
        self.eol_combo.addItems(["Без окончания", "CR (\\r)", "LF (\\n)", "CR+LF (\\r\\n)"])
        self.eol_combo.setCurrentText("CR (\\r)")
        self.eol_combo.setMaximumWidth(120)
        layout.addWidget(self.eol_combo)
    
    def send_command(self):
        """Отправить команду"""
        command = self.input_field.text()
        if not command:
            return
        
        if self.upper_check.isChecked():
            command = command.upper()
        
        # Определяем формат отправки
        is_binary = False
        command_to_send = command
        
        # Проверяем активный эмулятор
        if (self.parent.emulator_manager.current_emulator == "MODBUS_RTU" and 
            all(c in "0123456789ABCDEFabcdef" for c in command.replace(" ", ""))):
            try:
                command_to_send = bytes.fromhex(command.replace(" ", ""))
                is_binary = True
            except:
                pass
        
        if not is_binary:
            # Добавляем окончание строки
            eol_text = self.eol_combo.currentText()
            if eol_text == "CR (\\r)":
                command_to_send = command + "\r"
            elif eol_text == "LF (\\n)":
                command_to_send = command + "\n"
            elif eol_text == "CR+LF (\\r\\n)":
                command_to_send = command + "\r\n"
        
        # Отправляем
        if is_binary:
            self.parent.serial_manager.send_binary(command_to_send)
        else:
            self.parent.serial_manager.send_text(command_to_send)
        
        # Логируем
        from datetime import datetime
        timestamp = get_timestamp(self.parent)
        if is_binary:
            hex_str = command_to_send.hex().upper()
            self.parent.terminal.append(f"{timestamp}TX (HEX): {hex_str}")
        else:
            self.parent.terminal.append(f"{timestamp}TX: {command}")
        
        # Очищаем поле
        self.input_field.clear()