"""
РАБОЧИЙ COM ТЕРМИНАЛ - проверено работает
"""
import sys
import serial
import threading
import time
from datetime import datetime
from PySide6.QtWidgets import *
from PySide6.QtCore import *
from PySide6.QtGui import *

class WorkingTerminal(QMainWindow):
    def __init__(self):
        super().__init__()
        self.serial_port = None
        self.read_thread = None
        self.running = False
        self.rx_count = 0
        self.tx_count = 0
        self.data_buffer = ""
        
        self.setup_ui()
        self.scan_ports()
        
        print("[INIT] Терминал инициализирован")
        
    def setup_ui(self):
        self.setWindowTitle("Рабочий COM Терминал")
        self.setGeometry(100, 100, 1000, 800)
        
        # Центральный виджет
        central = QWidget()
        self.setCentralWidget(central)
        layout = QVBoxLayout(central)
        layout.setContentsMargins(5, 5, 5, 5)
        layout.setSpacing(5)
        
        # === ПАНЕЛЬ УПРАВЛЕНИЯ ===
        control_frame = QFrame()
        control_frame.setMaximumHeight(50)
        control_layout = QHBoxLayout(control_frame)
        control_layout.setContentsMargins(5, 5, 5, 5)
        
        # Порт
        self.port_combo = QComboBox()
        self.port_combo.setMinimumWidth(180)
        control_layout.addWidget(QLabel("Порт:"))
        control_layout.addWidget(self.port_combo)
        
        # Скорость
        self.baud_combo = QComboBox()
        self.baud_combo.addItems(["9600", "38400", "115200", "57600", "19200"])
        self.baud_combo.setCurrentText("9600")
        self.baud_combo.setMinimumWidth(100)
        control_layout.addWidget(QLabel("Скорость:"))
        control_layout.addWidget(self.baud_combo)
        
        # Кнопки
        self.connect_btn = QPushButton("📌 Подключить")
        self.connect_btn.clicked.connect(self.toggle_connection)
        self.connect_btn.setFixedWidth(120)
        control_layout.addWidget(self.connect_btn)
        
        self.refresh_btn = QPushButton("🔄 Обновить")
        self.refresh_btn.clicked.connect(self.scan_ports)
        self.refresh_btn.setFixedWidth(100)
        control_layout.addWidget(self.refresh_btn)
        
        self.clear_btn = QPushButton("🗑 Очистить")
        self.clear_btn.clicked.connect(self.clear_terminal)
        self.clear_btn.setFixedWidth(100)
        control_layout.addWidget(self.clear_btn)
        
        control_layout.addStretch()
        layout.addWidget(control_frame)
        
        # === ТЕРМИНАЛ ===
        self.terminal = QTextEdit()
        self.terminal.setReadOnly(True)
        self.terminal.setFont(QFont("Consolas", 9))
        self.terminal.setStyleSheet("""
            QTextEdit {
                background-color: #0a1929;
                color: #e6f7ff;
                border: 1px solid #555;
            }
        """)
        layout.addWidget(self.terminal, 1)
        
        # === ПАНЕЛЬ ОТПРАВКИ ===
        send_frame = QFrame()
        send_frame.setMaximumHeight(50)
        send_layout = QHBoxLayout(send_frame)
        send_layout.setContentsMargins(5, 5, 5, 5)
        
        self.input_field = QLineEdit()
        self.input_field.setPlaceholderText("Введите команду и нажмите Enter...")
        self.input_field.returnPressed.connect(self.send_command)
        send_layout.addWidget(self.input_field)
        
        self.send_btn = QPushButton("📤 Отправить")
        self.send_btn.clicked.connect(self.send_command)
        self.send_btn.setFixedWidth(120)
        send_layout.addWidget(self.send_btn)
        
        # Чекбокс верхнего регистра
        self.upper_check = QCheckBox("ВВЕРХ")
        self.upper_check.setToolTip("Автоматический перевод в верхний регистр")
        send_layout.addWidget(self.upper_check)
        
        # Чекбокс автопрокрутки
        self.autoscroll_check = QCheckBox("Автопрокрутка")
        self.autoscroll_check.setChecked(True)
        send_layout.addWidget(self.autoscroll_check)
        
        layout.addWidget(send_frame)
        
        # === СТАТУСНАЯ СТРОКА ===
        self.status_bar = QStatusBar()
        self.setStatusBar(self.status_bar)
        
        self.rx_label = QLabel("RX: 0")
        self.tx_label = QLabel("TX: 0")
        self.time_label = QLabel()
        self.status_label = QLabel("Готово")
        
        self.status_bar.addWidget(self.rx_label)
        self.status_bar.addWidget(QLabel(" | "))
        self.status_bar.addWidget(self.tx_label)
        self.status_bar.addWidget(QLabel(" | "))
        self.status_bar.addWidget(self.time_label)
        self.status_bar.addWidget(QLabel(" | "))
        self.status_bar.addWidget(self.status_label)
        
        # Таймер для обновления времени
        self.timer = QTimer()
        self.timer.timeout.connect(self.update_time)
        self.timer.start(1000)
        self.update_time()
        
        print("[UI] Интерфейс создан")
    
    def update_time(self):
        """Обновление времени"""
        current_time = datetime.now().strftime("%H:%M:%S")
        self.time_label.setText(current_time)
    
    def scan_ports(self):
        """Сканирование доступных портов"""
        import serial.tools.list_ports
        
        self.port_combo.clear()
        ports = list(serial.tools.list_ports.comports())
        
        for port in ports:
            self.port_combo.addItem(f"{port.device} - {port.description}", port.device)
        
        if ports:
            self.status_label.setText(f"Найдено портов: {len(ports)}")
            print(f"[SCAN] Найдено {len(ports)} портов")
        else:
            self.port_combo.addItem("Нет доступных портов")
            self.status_label.setText("Порты не найдены")
            print("[SCAN] Порты не найдены")
    
    def toggle_connection(self):
        """Подключение/отключение"""
        if self.serial_port and self.serial_port.is_open:
            self.disconnect_serial()
        else:
            self.connect_serial()
    
    def connect_serial(self):
        """Подключение к последовательному порту"""
        try:
            port_name = self.port_combo.currentData()
            baud_rate = int(self.baud_combo.currentText())
            
            if not port_name or port_name == "Нет доступных портов":
                self.log_message("[ОШИБКА] Выберите порт")
                return
            
            print(f"[CONNECT] Подключение к {port_name} на {baud_rate} бод")
            
            # Открываем порт с правильными параметрами
            self.serial_port = serial.Serial(
                port=port_name,
                baudrate=baud_rate,
                bytesize=serial.EIGHTBITS,
                parity=serial.PARITY_NONE,
                stopbits=serial.STOPBITS_ONE,
                timeout=0.1,  # Важно: небольшой таймаут для неблокирующего чтения
                write_timeout=1.0,
                xonxoff=False,
                rtscts=False,
                dsrdtr=False
            )
            
            # Сбрасываем буферы
            self.serial_port.reset_input_buffer()
            self.serial_port.reset_output_buffer()
            
            # Запускаем поток чтения
            self.running = True
            self.read_thread = threading.Thread(target=self.read_serial_thread, daemon=True)
            self.read_thread.start()
            
            # Обновляем UI
            self.connect_btn.setText("🔌 Отключить")
            self.status_label.setText(f"Подключено к {port_name}")
            self.log_message(f"[{datetime.now().strftime('%H:%M:%S')}] ✓ Подключено к {port_name} ({baud_rate} бод)")
            
            print(f"[CONNECT] Успешно подключено к {port_name}")
            
        except serial.SerialException as e:
            error_msg = str(e)
            self.log_message(f"[ОШИБКА] Не удалось подключиться: {error_msg}")
            self.status_label.setText("Ошибка подключения")
            print(f"[CONNECT ERROR] {error_msg}")
            
            # Сбрасываем состояние
            if self.serial_port:
                try:
                    self.serial_port.close()
                except:
                    pass
                self.serial_port = None
                
        except Exception as e:
            self.log_message(f"[ОШИБКА] {str(e)}")
            print(f"[CONNECT ERROR] {e}")
    
    def disconnect_serial(self):
        """Отключение от порта"""
        print("[DISCONNECT] Отключение...")
        
        # Останавливаем поток чтения
        self.running = False
        
        # Закрываем порт
        if self.serial_port and self.serial_port.is_open:
            try:
                self.serial_port.close()
                print("[DISCONNECT] Порт закрыт")
            except Exception as e:
                print(f"[DISCONNECT ERROR] {e}")
        
        self.serial_port = None
        
        # Ждем завершения потока
        if self.read_thread and self.read_thread.is_alive():
            self.read_thread.join(timeout=1.0)
        
        # Обновляем UI
        self.connect_btn.setText("📌 Подключить")
        self.status_label.setText("Отключено")
        self.log_message(f"[{datetime.now().strftime('%H:%M:%S')}] Отключено")
        
        print("[DISCONNECT] Успешно отключено")
    
    def read_serial_thread(self):
        """Поток чтения данных из порта"""
        print("[READ THREAD] Запуск потока чтения")
        
        buffer = bytearray()
        
        while self.running and self.serial_port and self.serial_port.is_open:
            try:
                # Проверяем доступные данные
                if self.serial_port.in_waiting > 0:
                    # Читаем ВСЕ доступные данные
                    data = self.serial_port.read(self.serial_port.in_waiting)
                    
                    if data:
                        buffer.extend(data)
                        
                        # Обрабатываем буфер, ища концы строк
                        while True:
                            # Ищем \r или \n
                            if b'\r' in buffer:
                                idx = buffer.index(b'\r')
                            elif b'\n' in buffer:
                                idx = buffer.index(b'\n')
                            else:
                                break  # Нет конца строки
                            
                            # Извлекаем строку
                            line_bytes = buffer[:idx]
                            buffer = buffer[idx+1:]  # Убираем символ конца строки
                            
                            # Декодируем
                            try:
                                line = line_bytes.decode('utf-8', errors='ignore')
                            except:
                                try:
                                    line = line_bytes.decode('cp1251', errors='ignore')
                                except:
                                    line = line_bytes.decode('latin-1', errors='ignore')
                            
                            # Отправляем в UI если строка не пустая
                            line = line.strip()
                            if line:
                                # Используем сигнал для обновления UI из главного потока
                                QMetaObject.invokeMethod(self, "add_received_data", 
                                                        Qt.QueuedConnection,
                                                        Q_ARG(str, line))
                
                # Обработка оставшегося буфера (если накопилось много данных без \r\n)
                if len(buffer) > 100:  # 100 байт без конца строки
                    try:
                        line = buffer.decode('utf-8', errors='ignore').strip()
                        if line:
                            QMetaObject.invokeMethod(self, "add_received_data",
                                                    Qt.QueuedConnection,
                                                    Q_ARG(str, line))
                        buffer.clear()
                    except:
                        buffer.clear()
                
                # Небольшая пауза чтобы не грузить CPU
                time.sleep(0.01)
                
            except Exception as e:
                print(f"[READ THREAD ERROR] {e}")
                time.sleep(0.1)
        
        print("[READ THREAD] Поток чтения завершен")
    
    @Slot(str)
    def add_received_data(self, data: str):
        """Добавление полученных данных в терминал (вызывается из главного потока)"""
        if not data:
            return
        
        # Обновляем счетчик
        self.rx_count += 1
        self.rx_label.setText(f"RX: {self.rx_count}")
        
        # Добавляем в терминал с временной меткой
        timestamp = datetime.now().strftime("%H:%M:%S")
        self.terminal.append(f"[{timestamp}] RX: {data}")
        
        # Автопрокрутка
        if self.autoscroll_check.isChecked():
            cursor = self.terminal.textCursor()
            cursor.movePosition(QTextCursor.End)
            self.terminal.setTextCursor(cursor)
            self.terminal.ensureCursorVisible()
        
        print(f"[DATA] Получено: {repr(data)}")
    
    def send_command(self):
        """Отправка команды"""
        command = self.input_field.text()
        if not command:
            return
        
        # Перевод в верхний регистр если нужно
        if self.upper_check.isChecked():
            command = command.upper()
        
        # Добавляем \r\n для большинства устройств
        command_to_send = command + "\r\n"
        
        try:
            if self.serial_port and self.serial_port.is_open:
                # Отправляем данные
                self.serial_port.write(command_to_send.encode('utf-8'))
                self.serial_port.flush()  # Ожидаем отправки
                
                # Обновляем счетчик
                self.tx_count += 1
                self.tx_label.setText(f"TX: {self.tx_count}")
                
                # Логируем отправку
                timestamp = datetime.now().strftime("%H:%M:%S")
                self.terminal.append(f"[{timestamp}] TX: {command}")
                
                # Автопрокрутка
                if self.autoscroll_check.isChecked():
                    cursor = self.terminal.textCursor()
                    cursor.movePosition(QTextCursor.End)
                    self.terminal.setTextCursor(cursor)
                    self.terminal.ensureCursorVisible()
                
                # Очищаем поле ввода (опционально)
                # self.input_field.clear()
                
                print(f"[SEND] Отправлено: {repr(command)}")
                
            else:
                self.log_message("[ОШИБКА] Не подключено к порту")
                
        except Exception as e:
            error_msg = str(e)
            self.log_message(f"[ОШИБКА отправки] {error_msg}")
            print(f"[SEND ERROR] {error_msg}")
    
    def log_message(self, message: str):
        """Добавление служебного сообщения"""
        self.terminal.append(message)
        
        if self.autoscroll_check.isChecked():
            cursor = self.terminal.textCursor()
            cursor.movePosition(QTextCursor.End)
            self.terminal.setTextCursor(cursor)
    
    def clear_terminal(self):
        """Очистка терминала"""
        self.terminal.clear()
        self.log_message(f"[{datetime.now().strftime('%H:%M:%S')}] Терминал очищен")
    
    def closeEvent(self, event):
        """Обработка закрытия окна"""
        print("[CLOSE] Закрытие приложения...")
        
        # Отключаемся от порта
        if self.serial_port and self.serial_port.is_open:
            self.disconnect_serial()
        
        # Останавливаем таймер
        if self.timer.isActive():
            self.timer.stop()
        
        event.accept()
        print("[CLOSE] Приложение закрыто")

def main():
    """Точка входа"""
    app = QApplication(sys.argv)
    
    # Настройки приложения
    app.setApplicationName("Рабочий COM Терминал")
    app.setStyle("Fusion")
    
    # Создаем и показываем окно
    window = WorkingTerminal()
    window.show()
    
    # Запускаем приложение
    sys.exit(app.exec())

if __name__ == "__main__":
    main()